-- Author: U_BMP
-- Group: https://vk.com/biomodprod_utilit_fs
-- Date: 19.11.2025

FieldGroundMudPhysics = {}
FieldGroundMudPhysics.name = g_currentModName or "FieldGroundMudPhysics"
FieldGroundMudPhysics.path = g_currentModDirectory or ""
addModEventListener(FieldGroundMudPhysics)

-- =========================================================
-- НАСТРОЙКИ (ГЛОБАЛЬНЫЕ)
-- =========================================================
FieldGroundMudPhysics.enabled = true        -- включение / выключение всей системы грязи
FieldGroundMudPhysics.debug   = false       -- режим отладки (логи в консоль)

FieldGroundMudPhysics.mpRadiusSyncEnable      = false
FieldGroundMudPhysics.mpRadiusSyncIntervalMs  = 120

-- ---------------------------------------------------------
-- ВЛАЖНОСТЬ И ДОЖДЬ
-- ---------------------------------------------------------
FieldGroundMudPhysics.wetnessThreshold     = 0.12  -- минимальная влажность, при которой грязь начинает работать
FieldGroundMudPhysics.rainForcesWetnessMin = 0.55  -- минимальная эффективная влажность при активном дожде
FieldGroundMudPhysics.rainScaleThreshold   = 0.04  -- порог интенсивности дождя, после которого он "принудительно" мочит землю

-- ---------------------------------------------------------
-- ПЛАВНОЕ ВКЛЮЧЕНИЕ / ВЫКЛЮЧЕНИЕ ЭФФЕКТА
-- ---------------------------------------------------------
FieldGroundMudPhysics.rampInSec  = 8.0      -- время (сек), за которое грязь полностью "набирается"
FieldGroundMudPhysics.rampOutSec = 3.2      -- время (сек), за которое эффект грязи сходит
FieldGroundMudPhysics.rampPow    = 1.55     -- степень кривой (больше = резче в конце)

-- ---------------------------------------------------------
-- МЕТР ЗАРЫВАНИЯ ТЕХНИКИ (0..1)
-- ---------------------------------------------------------
FieldGroundMudPhysics.sinkInSpeed    = 2.25 -- скорость увеличения "утопания" техники
FieldGroundMudPhysics.sinkOutSpeed   = 0.85 -- скорость выхода из утопания
FieldGroundMudPhysics.stuckThreshold = 0.82 -- порог, после которого техника считается застрявшей

-- ---------------------------------------------------------
-- ОГРАНИЧЕНИЕ СКОРОСТИ В ГРЯЗИ
-- ---------------------------------------------------------
FieldGroundMudPhysics.maxSpeedMudKph   = 14.0 -- максимальная скорость в грязи (км/ч)
FieldGroundMudPhysics.maxSpeedStuckKph = 3.5  -- максимальная скорость при застревании

-- ---------------------------------------------------------
-- НАГРУЗКА НА ДВИГАТЕЛЬ / ПОТЕРЯ МОЩНОСТИ
-- ---------------------------------------------------------
FieldGroundMudPhysics.motorLoadEnable   = true -- включить дополнительную нагрузку на двигатель
FieldGroundMudPhysics.motorLoadFromSink = 2.6  -- влияние глубины зарывания на нагрузку
FieldGroundMudPhysics.motorLoadFromSlip = 1.6  -- влияние пробуксовки на нагрузку
FieldGroundMudPhysics.motorLoadMaxMult  = 14.0 -- максимальный множитель нагрузки
FieldGroundMudPhysics.motorLoadMinEff   = 0.10 -- минимальная эффективность грязи для применения нагрузки

-- ---------------------------------------------------------
-- ВЯЗКОЕ СОПРОТИВЛЕНИЕ КАЧЕНИЮ КОЛЁС (АНАЛОГ "ТОРМОЗА")
-- ---------------------------------------------------------
FieldGroundMudPhysics.wheelBrakeEnable    = true  -- включить вязкое сопротивление колёс
FieldGroundMudPhysics.wheelBrakeBase      = 3.228 -- базовое сопротивление
FieldGroundMudPhysics.wheelBrakeFromSink  = 6.252 -- вклад глубины зарывания
FieldGroundMudPhysics.wheelBrakeFromSlip  = 3.0   -- вклад пробуксовки
FieldGroundMudPhysics.wheelBrakeRatio     = 0.034 -- коэффициент масштабирования
FieldGroundMudPhysics.wheelBrakeMaxRatio  = 2.40  -- максимальный множитель сопротивления
FieldGroundMudPhysics.wheelBrakeMinEff    = 0.08  -- минимальная эффективность грязи
FieldGroundMudPhysics.wheelBrakeSmoothIn  = 2.1   -- скорость нарастания сопротивления
FieldGroundMudPhysics.wheelBrakeSmoothOut = 6.0   -- скорость спада сопротивления

-- ---------------------------------------------------------
-- РЕАЛЬНОЕ УМЕНЬШЕНИЕ РАДИУСА КОЛЕСА (ПОГРУЖЕНИЕ В ГРУНТ)
-- ---------------------------------------------------------
FieldGroundMudPhysics.radiusSinkEnable   = true  -- включить уменьшение радиуса колеса
FieldGroundMudPhysics.radiusMinFactor    = 0.57  -- минимальный радиус (доля от оригинального)
FieldGroundMudPhysics.radiusSinkInSpeed  = 0.10 -- скорость погружения колеса
FieldGroundMudPhysics.radiusSinkOutSpeed = 0.068  -- скорость восстановления радиуса

-- ---------------------------------------------------------
-- УПРАВЛЕНИЕ СЦЕПЛЕНИЕМ (profile.slip)
-- ---------------------------------------------------------
FieldGroundMudPhysics.slipMinMul = 0.58  -- минимальный множитель сцепления (больше буксует)
FieldGroundMudPhysics.slipMaxMul = 0.98  -- максимальный множитель сцепления

-- ---------------------------------------------------------
-- "ЖИВАЯ ЗЕМЛЯ" (СЛУЧАЙНЫЕ МИКРО-ПОДЪЁМЫ КОЛЕСА В ГРЯЗИ)
-- ---------------------------------------------------------
FieldGroundMudPhysics.livingSoilEnable        = true   -- включить эффект "дышащей" земли
FieldGroundMudPhysics.livingSoilSpeedMinKph   = 2.5    -- минимальная скорость, ниже не срабатывает
FieldGroundMudPhysics.livingSoilSlipMax       = 0.06   -- при сильной буксовке эффект отключён
FieldGroundMudPhysics.livingSoilDeepFrac      = 0.80   -- считаем колесо глубоко утонувшим (>80%)
FieldGroundMudPhysics.livingSoilPulseMinFrac  = 0.004  -- минимальный импульс (доля радиуса)
FieldGroundMudPhysics.livingSoilPulseMaxFrac  = 0.015  -- максимальный импульс
FieldGroundMudPhysics.livingSoilIntervalMin   = 0.8    -- минимум секунд между импульсами
FieldGroundMudPhysics.livingSoilIntervalMax   = 2.2    -- максимум секунд между импульсами
FieldGroundMudPhysics.livingSoilChance        = 0.55   -- шанс срабатывания импульса (0..1)

-- ---------------------------------------------------------
-- ПРЕДОТВРАЩЕНИЕ "ВЫТАЛКИВАНИЯ" КОЛЕС ИЗ ГРЯЗИ
-- ---------------------------------------------------------
FieldGroundMudPhysics.radiusRecoverInMudSpeed     = 0.02 -- восстановление радиуса, пока всё ещё в грязи
FieldGroundMudPhysics.radiusRecoverOutOfMudSpeed  = 0.95 -- восстановление радиуса после выезда на твёрдое

-- ---------------------------------------------------------
-- АНТИ-ДРОЖАНИЕ ПРИ ОСТАНОВКЕ
-- ---------------------------------------------------------
FieldGroundMudPhysics.freezeRadiusWhenStopped = true   -- замораживать радиус при остановке
FieldGroundMudPhysics.freezeStopSpeedKph      = 0.60   -- скорость, ниже которой считаем остановкой
FieldGroundMudPhysics.freezeStopSlip          = 0.010  -- допустимая пробуксовка при остановке
FieldGroundMudPhysics.freezeSettleSeconds     = 0.45   -- время до полной фиксации радиуса
FieldGroundMudPhysics.freezeRadiusEps         = 0.0015 -- минимальный шаг изменения радиуса

-- Максимальное восстановление радиуса на месте (в метрах)
FieldGroundMudPhysics.stopRecoverRadiusMaxM = 0.003

-- ---------------------------------------------------------
-- ОСЕДАНИЕ НА МЕСТЕ (НО БЕЗ БЕСКОНЕЧНОГО УТОПАНИЯ)
-- ---------------------------------------------------------
FieldGroundMudPhysics.stopSettleEnable        = true   -- разрешить оседание на месте
FieldGroundMudPhysics.stopSettleMaxExtraFrac  = 0.035  -- максимум оседания (доля радиуса)
FieldGroundMudPhysics.stopSettleSlipAllow     = 0.080  -- если slip выше — считаем что техника "едет"
FieldGroundMudPhysics.stopSettleSpeedKph      = 1.80   -- скорость, ниже которой считаем стоянкой
FieldGroundMudPhysics.stopSettleInSpeed       = 0.22   -- скорость оседания на месте

-- ---------------------------------------------------------
-- РЕДКОЕ "НАМЕРТВО ЗАСТРЯТЬ" (ГЛУБОКАЯ ГРЯЗЬ)
-- ---------------------------------------------------------
FieldGroundMudPhysics.permaStuckEnable           = true   -- включить перманентное застревание
FieldGroundMudPhysics.permaStuckChanceOnStruggle = 0.020  -- шанс в секунду при борьбе
FieldGroundMudPhysics.permaStuckSlipMin          = 0.12   -- минимальный slip для активации
FieldGroundMudPhysics.permaStuckSinkMin          = 0.38   -- минимальная глубина зарывания
FieldGroundMudPhysics.permaStuckRadiusFactor     = 0.40   -- минимальный радиус колеса
FieldGroundMudPhysics.permaStuckCooldownMin      = 45.0   -- минимальный кулдаун (сек)
FieldGroundMudPhysics.permaStuckCooldownMax      = 120.0  -- максимальный кулдаун
FieldGroundMudPhysics.permaStuckRampInSec        = 3.5    -- время "затягивания"
FieldGroundMudPhysics.permaStuckRampOutSec       = 0.60   -- время выхода

-- ---------------------------------------------------------
-- РАСПРОСТРАНЕНИЕ ЗАСТРЕВАНИЯ НА ВСЮ МАШИНУ
-- ---------------------------------------------------------
FieldGroundMudPhysics.permaStuckSpreadEnable    = true  -- включить распространение
FieldGroundMudPhysics.permaStuckSpreadInSec     = 3.1   -- за сколько секунд тянем остальные колёса
FieldGroundMudPhysics.permaStuckSpreadOnlyInMud = true  -- только если колесо в грязном профиле
FieldGroundMudPhysics.permaStuckBrakeBoost      = 0.58  -- доп. сопротивление колёсам (+55%)
FieldGroundMudPhysics.permaStuckMinWheelsInMud  = 2     -- минимум колёс в грязи для активации

-- ---------------------------------------------------------
-- ДОПОЛНИТЕЛЬНЫЕ ЧАСТИЦЫ ГРЯЗИ
-- ---------------------------------------------------------
FieldGroundMudPhysics.extraParticlesEnable      = true   -- включить дополнительные частицы
FieldGroundMudPhysics.extraParticlesI3D         = "particleSystems/mudField.i3d"      -- файл частиц
FieldGroundMudPhysics.extraEmitterShapeI3D      = "particleSystems/mudEmitShape.i3d"  -- форма эмиттера
FieldGroundMudPhysics.extraParticleClipDist     = 90     -- дистанция отрисовки (не работает)
FieldGroundMudPhysics.extraParticleMaxCount     = 900    -- максимум частиц (не работает)
FieldGroundMudPhysics.extraParticleOffsetY      = -0.22  -- вертикальный сдвиг
FieldGroundMudPhysics.extraParticleMinSpeedKph  = 18.0   -- минимальная скорость для эффекта

-- ---------------------------------------------------------
-- КОЛЕИ (ДЕФОРМАЦИЯ ТЕРРЕЙНА)
-- ---------------------------------------------------------
FieldGroundMudPhysics.rutsEnable        = true  -- включить колеи
FieldGroundMudPhysics.rutsIntervalMs    = 180   -- интервал между "копанием" одним колесом
FieldGroundMudPhysics.rutsMaxDepthM     = 0.19  -- максимальная глубина колеи (м)
FieldGroundMudPhysics.rutsRadiusMul     = 0.69  -- радиус кисти от ширины колеса
FieldGroundMudPhysics.rutsHardness      = 0.52  -- мягкость кисти (0..1)
FieldGroundMudPhysics.rutsMinSinkFrac   = 0.16  -- минимальное утопание для рисования колеи
FieldGroundMudPhysics.rutsSlipBonusMul  = 0.75  -- усиление от пробуксовки
FieldGroundMudPhysics.rutsSpeedMinKph   = 0.8   -- если почти стоим — не копаем

-- ---------------------------------------------------------
-- НАЛИПАНИЕ ГРЯЗИ НА ТЕХНИКУ (WASHABLE)
-- ---------------------------------------------------------
FieldGroundMudPhysics.dirtEnable        = true  -- включить загрязнение
FieldGroundMudPhysics.dirtMinEffMud     = 0.06  -- минимальная эффективность грязи
FieldGroundMudPhysics.dirtWetnessMin    = 0.18  -- минимальная влажность для налипания
FieldGroundMudPhysics.dirtBodyPerSec    = 0.010 -- загрязнение кузова в сек
FieldGroundMudPhysics.dirtWheelPerSec   = 0.055 -- загрязнение колёс в сек
FieldGroundMudPhysics.dirtSpeedBoostKph = 16.0  -- ускорение загрязнения от скорости
FieldGroundMudPhysics.dirtMax           = 1.00  -- максимальная грязь

-- ---------------------------------------------------------
-- ОБЩЕЕ ЗАРЫВАНИЕ КОЛЁС (АНТИ-"ПЛАВАЮЩИЕ" КОЛЁСА)
-- ---------------------------------------------------------
FieldGroundMudPhysics.sharedSinkEnable           = true  -- включить общее зарывание
FieldGroundMudPhysics.sharedSinkMinFrac          = 0.90  -- доля от максимального зарывания
FieldGroundMudPhysics.sharedSinkNoContactBoost   = 1.00  -- усиление, если колесо без контакта

-- ---------------------------------------------------------
-- УДАЛЕНИЕ УРОЖАЯ (fruitType) В НОЛЬ ПОД КОЛЕСОМ В ГРЯЗИ
-- ---------------------------------------------------------
FieldGroundMudPhysics.eraseFruitEnable      = false
FieldGroundMudPhysics.eraseFoliageOnlyFields = true -- true = стираем foliage только на полях (groundType), false = везде
FieldGroundMudPhysics.eraseFruitMinEffMud   = 0.18   -- минимум effMud, чтобы удалять
FieldGroundMudPhysics.eraseFruitIntervalMs  = 220    -- троттлинг на wheelPhysics
FieldGroundMudPhysics.eraseFruitSpeedMinKph = 1.0
FieldGroundMudPhysics.eraseFruitDoVanillaWheelDestructionFirst = true -- сначала ваниль crushed, потом 0


-- =========================================================
-- FIELD GROUND PROFILES
--
-- name              : имя типа земли (для дебага/логов)
-- mud (0..1)        : базовая вязкость земли (чем больше — тем грязнее)
-- wetMul            : насколько дождь/влажность усиливает грязь
-- sinkMul           : множитель скорости и глубины зарывания техники
-- brakeMul          : усиление вязкого сопротивления качению колёс
-- motorMul          : дополнительная нагрузка на двигатель (потеря тяги)
-- radiusMinFactor   : минимальный радиус колеса (доля от оригинального),
--                     чем меньше — тем глубже колесо может утонуть
-- fxExtra           : включает дополнительные грязевые частицы (визуал)
-- permaStuck        : разрешает редкое "намертво застрять" на этом типе земли
--
-- slip (0.60..1.15) : множитель сцепления колеса с грунтом (через tire friction)
--                     меньше 1.0 = больше пробуксовка, больше 1.0 = меньше пробуксовка

-- =========================================================

FieldGroundMudPhysics.groundProfiles = {
    [1]  = { name="StubbleTillage", mud=0.32, wetMul=1.10, sinkMul=7.05, brakeMul=6.05, motorMul=1.05, radiusMinFactor=0.52, dirtMul=1.08, slip=0.85, fxExtra=true, permaStuck=true },
    [2]  = { name="Cultivated",     mud=0.48, wetMul=1.28, sinkMul=7.10, brakeMul=6.10, motorMul=1.08, radiusMinFactor=0.46, dirtMul=1.28, slip=0.80, fxExtra=true, permaStuck=true },
    [3]  = { name="Seedbed",        mud=0.30, wetMul=1.08, sinkMul=7.05, brakeMul=6.06, motorMul=1.05, radiusMinFactor=0.53, dirtMul=1.05, slip=0.87, fxExtra=true, permaStuck=true },
    [4]  = { name="Plowed",         mud=0.70, wetMul=1.55, sinkMul=9.55, brakeMul=8.25, motorMul=1.42, radiusMinFactor=0.42, dirtMul=1.60, slip=0.88, fxExtra=true, permaStuck=true },
    [5]  = { name="RolledSeedbed",  mud=0.16, wetMul=0.85, sinkMul=6.85, brakeMul=5.90, motorMul=0.92, radiusMinFactor=0.58, dirtMul=0.80, slip=0.96, fxExtra=true, permaStuck=true },
    [6]  = { name="Ridge",          mud=0.40, wetMul=1.20, sinkMul=7.12, brakeMul=6.02, motorMul=1.10, radiusMinFactor=0.50, dirtMul=1.15, slip=0.90, fxExtra=true, permaStuck=true },
    [7]  = { name="Sown",           mud=0.14, wetMul=0.78, sinkMul=6.80, brakeMul=5.85, motorMul=0.88, radiusMinFactor=0.60, dirtMul=0.72, slip=0.98, fxExtra=true, permaStuck=true },
    [8]  = { name="DirectSown",     mud=0.16, wetMul=0.82, sinkMul=6.85, brakeMul=5.88, motorMul=0.90, radiusMinFactor=0.59, dirtMul=0.78, slip=0.95, fxExtra=true, permaStuck=true },
    [9]  = { name="Planted",        mud=0.22, wetMul=0.95, sinkMul=6.95, brakeMul=5.95, motorMul=0.96, radiusMinFactor=0.56, dirtMul=0.92, slip=0.99, fxExtra=true, permaStuck=true },
    [10] = { name="RidgeSown",      mud=0.32, wetMul=1.10, sinkMul=7.05, brakeMul=6.05, motorMul=1.04, radiusMinFactor=0.53, dirtMul=1.06, slip=0.96, fxExtra=true, permaStuck=true },
    [11] = { name="Rollerlines",    mud=0.08, wetMul=0.60, sinkMul=6.65, brakeMul=5.70, motorMul=0.80, radiusMinFactor=0.66, dirtMul=0.55, slip=0.98, fxExtra=true, permaStuck=true },
    [12] = { name="HarvestReady",   mud=0.20, wetMul=0.92, sinkMul=6.92, brakeMul=5.92, motorMul=0.94, radiusMinFactor=0.57, dirtMul=0.90, slip=0.92, fxExtra=true, permaStuck=true },
    [13] = { name="HarvestReadyO",  mud=0.18, wetMul=0.88, sinkMul=6.90, brakeMul=5.90, motorMul=0.92, radiusMinFactor=0.58, dirtMul=0.86, slip=0.93, fxExtra=true, permaStuck=true },
	[14] = { name="Grass",  		mud=0.10, wetMul=0.88, sinkMul=6.90, brakeMul=5.90, motorMul=0.92, radiusMinFactor=0.78, dirtMul=0.86, slip=0.83, fxExtra=true, permaStuck=false },
	[15] = { name="GrassCut)", 		mud=0.10, wetMul=0.88, sinkMul=6.90, brakeMul=5.90, motorMul=0.92, radiusMinFactor=0.78, dirtMul=0.86, slip=0.83, fxExtra=true, permaStuck=false },
}

-- =========================================================
-- UTILS
-- =========================================================
local computeWetnessEffective

local function clamp(v,a,b)
    if v < a then return a end
    if v > b then return b end
    return v
end

local function mpsToKph(v) return (v or 0) * 3.6 end
local function kphToMps(v) return (v or 0) / 3.6 end


local getState

local function fgEnsureWheelIndices()
    if g_currentMission == nil or g_currentMission.vehicles == nil then return end
    for _, v in pairs(g_currentMission.vehicles) do
        if v ~= nil and v.spec_wheels ~= nil and v.spec_wheels.wheels ~= nil then
            for i, w in ipairs(v.spec_wheels.wheels) do
                if w ~= nil then
                    w.wheelIndex = i
                end
            end
        end
    end
end

-- =========================================================

function FieldGroundMudPhysics:getWheelNoContactFlag(wp)
    if wp == nil then return false end
    if wp.hasGroundContact ~= nil then
        return wp.hasGroundContact == false
    end
    if wp.hasSoilContact ~= nil then
        return wp.hasSoilContact == false
    end
    return false
end

function FieldGroundMudPhysics:computeVehicleSharedTargetExtra(vehicle, wetnessEff)
    if vehicle == nil or vehicle.spec_wheels == nil or vehicle.spec_wheels.wheels == nil then
        return 0, 0
    end

    local wheels = vehicle.spec_wheels.wheels
    local maxTarget, sumTarget, cnt = 0, 0, 0

    for _, w in ipairs(wheels) do
        if w ~= nil and w.physics ~= nil then
            local wp = w.physics

            local wx, _, wz = self:getWheelContactPos(w)
            if wx ~= nil then
                local mudBase, _, prof = self:getProfileAtWorldPos(wx, wz)
                if prof ~= nil and (mudBase or 0) > 0.01 then
                    local wetFactor = clamp(((wetnessEff or 0) - 0.15) / 0.55, 0, 1)
                    local wetMul    = prof.wetMul or 1.0
                    local effMud    = clamp((mudBase or 0) * wetFactor * wetMul, 0, 1)

                    local r0 = wp.__fgOrigRadius or wp.radiusOriginal or wp.radius
                    if type(r0) == "number" and r0 > 0.05 and r0 == r0 then
                        local minFactor = prof.radiusMinFactor or self.radiusMinFactor or 0.52
                        local minRadius = math.max(0.12, r0 * clamp(minFactor, 0.12, 0.98))
                        local maxExtra  = math.max(0, r0 - minRadius)

                        local stV = getState(vehicle)
                        local sinkMul = prof.sinkMul or 1.0
                        local slip = (wp.netInfo ~= nil and wp.netInfo.slip) or 0

                        local vSpeed = (vehicle.getLastSpeed and vehicle:getLastSpeed()) or vehicle.lastSpeedReal or 0
                        local speedKph = mpsToKph(vSpeed)

                        local moving01   = clamp(speedKph / 6.0, 0, 1)
                        local slip01     = clamp((slip or 0) / 0.18, 0, 1)
                        local moveFactor = math.max(moving01, slip01)

                        local k = clamp(effMud * sinkMul * (0.10 + moveFactor * (0.85 + (stV.sink or 0) * 0.55)), 0, 1)
                        local targetExtra = (r0 * 0.28) * k

                        targetExtra = clamp(targetExtra, 0, maxExtra)

                        maxTarget = math.max(maxTarget, targetExtra)
                        sumTarget = sumTarget + targetExtra
                        cnt = cnt + 1
                    end
                end
            end
        end
    end

    local avgTarget = (cnt > 0) and (sumTarget / cnt) or 0
    return maxTarget, avgTarget
end

function FieldGroundMudPhysics:getSharedTargetExtraCached(vehicle, wetnessEff)
    local stV = getState(vehicle)
    local t = g_time or 0

    if stV.__fgSharedTick ~= t then
        stV.__fgSharedTick = t
        local mx, av = self:computeVehicleSharedTargetExtra(vehicle, wetnessEff)
        stV.__fgSharedTargetMax = mx or 0
        stV.__fgSharedTargetAvg = av or 0
    end

    return stV.__fgSharedTargetMax or 0, stV.__fgSharedTargetAvg or 0
end

local function mpSafeDelete(node)
    if node ~= nil and node ~= 0 and entityExists(node) then
        delete(node)
    end
end

local function getWeather()
    if g_currentMission ~= nil and g_currentMission.environment ~= nil and g_currentMission.environment.weather ~= nil then
        return g_currentMission.environment.weather
    end
    return nil
end

computeWetnessEffective = function(groundWetness)
    local wetnessEff = groundWetness or 0
    local weather = getWeather()
    if weather ~= nil then
        if weather.getGroundWetness ~= nil then
            wetnessEff = math.max(wetnessEff, weather:getGroundWetness() or 0)
        end
        if weather.getRainFallScale ~= nil then
            local rain = weather:getRainFallScale() or 0
            if rain >= (FieldGroundMudPhysics.rainScaleThreshold or 0) then
                wetnessEff = math.max(wetnessEff, FieldGroundMudPhysics.rainForcesWetnessMin or wetnessEff)
            end
        end
    end
    return wetnessEff
end

function FieldGroundMudPhysics:isInShopPreview(vehicle)
    if vehicle == nil then return false end

    if vehicle.getIsInShowroom ~= nil then
        return vehicle:getIsInShowroom()
    end

    local ps = nil
    if vehicle.getPropertyState ~= nil then
        ps = vehicle:getPropertyState()
    else
        ps = vehicle.propertyState
    end

    return ps ~= nil and VehiclePropertyState ~= nil and ps == VehiclePropertyState.SHOP_CONFIG
end

function FieldGroundMudPhysics:getWheelContactPos(wheel)
    if wheel == nil then return nil end

    local node =
        wheel.driveNode or
        wheel.wheelNode or
        wheel.node or
        wheel.repr

    if node == nil or node == 0 then
        return nil
    end

    local x, y, z = getWorldTranslation(node)

    local r = 0.45
    if wheel.physics ~= nil then
        r = wheel.physics.radiusOriginal or wheel.physics.radius or r
    end
    r = wheel.radiusOriginal or wheel.radius or r

    return x, y - r, z
end

function FieldGroundMudPhysics:getWheelDestructionParallelogram(wheel)
    if wheel == nil then return nil end

    local repr = wheel.repr
    if repr == nil or repr == 0 then return nil end

    local driveNode = wheel.driveNode or wheel.node or wheel.wheelNode or repr
    if driveNode == nil or driveNode == 0 then return nil end

    local width = wheel.width or 0.5
    if wheel.physics ~= nil and type(wheel.physics.width) == "number" then
        width = wheel.physics.width
    end

    local halfW = math.max(0.10, (width * 0.5))

    local halfL = math.max(0.10, math.min(0.45, halfW))

    local lx, ly, lz = localToLocal(driveNode, repr, 0, 0, 0)

    local x0, _, z0 = localToWorld(repr, lx + halfW, ly, lz - halfL)
    local x1, _, z1 = localToWorld(repr, lx - halfW, ly, lz - halfL)
    local x2, _, z2 = localToWorld(repr, lx + halfW, ly, lz + halfL)

    return x0, z0, x1, z1, x2, z2
end

function FieldGroundMudPhysics:eraseFruitToZero(x0, z0, x1, z1, x2, z2)
    if g_server == nil then return end
    if g_currentMission == nil then return end
    if g_fruitTypeManager == nil or g_fruitTypeManager.getFruitTypes == nil then return end
    if g_terrainNode == nil or g_terrainNode == 0 then return end

    self._eraseFruitMM = self._eraseFruitMM or DensityMapMultiModifier.new()
    self._eraseFruitCache = self._eraseFruitCache or { mods = {}, filts = {}, fieldFilter = nil }

    if self._eraseFruitCache.fieldFilter == nil then
        if g_fieldGroundSystem ~= nil and g_fieldGroundSystem.getFieldFilter ~= nil then
            self._eraseFruitCache.fieldFilter = g_fieldGroundSystem:getFieldFilter()
        end
    end

    local mm = self._eraseFruitMM
    local cache = self._eraseFruitCache
    mm:reset()

    local fieldFilter = cache.fieldFilter

    local function keyOf(plane, start, chans)
        return tostring(plane) .. ":" .. tostring(start) .. ":" .. tostring(chans)
    end

    local function getOrCreate(plane, start, chans)
        if plane == nil or plane == 0 then return nil, nil end
        if type(start) ~= "number" or type(chans) ~= "number" or chans <= 0 then return nil, nil end

        local k = keyOf(plane, start, chans)
        local m = cache.mods[k]
        local f = cache.filts[k]

        if m == nil then
            m = DensityMapModifier.new(plane, start, chans, g_terrainNode)
            cache.mods[k] = m
        end
        if f == nil then
            f = DensityMapFilter.new(plane, start, chans)
            cache.filts[k] = f
        end
        return m, f
    end

    for _, ft in ipairs(g_fruitTypeManager:getFruitTypes()) do
        local m, f = getOrCreate(ft.terrainDataPlaneId, ft.startStateChannel, ft.numStateChannels)
        if m ~= nil and f ~= nil then
            m:setParallelogramWorldCoords(x0, z0, x1, z1, x2, z2, DensityCoordType.POINT_POINT_POINT)
            f:setValueCompareParams(DensityValueCompareType.GREATER, 0)

            if fieldFilter ~= nil then
                mm:addExecuteSet(0, m, f, fieldFilter)
            else
                mm:addExecuteSet(0, m, f)
            end
        end
    end

    mm:execute()
end

-- =========================================================
-- ERASE FOLIAGE CACHE
-- =========================================================
function FieldGroundMudPhysics:initEraseFoliageCache()
    if g_server == nil then return end
    if g_currentMission == nil then return end
    if g_fruitTypeManager == nil or g_fruitTypeManager.getFruitTypes == nil then return end
    if g_terrainNode == nil or g_terrainNode == 0 then return end

    self._eraseFoliageMM = self._eraseFoliageMM or DensityMapMultiModifier.new()

    self._eraseFoliageCache = self._eraseFoliageCache or {
        fruitMods   = {},
        fruitFilts  = {},
        haulmMods   = {},
        haulmFilts  = {},
        dynMods     = {},
        fieldFilter = nil
    }

    if self._eraseFoliageCache.fieldFilter == nil then
        if g_fieldGroundSystem ~= nil and g_fieldGroundSystem.getFieldFilter ~= nil then
            self._eraseFoliageCache.fieldFilter = g_fieldGroundSystem:getFieldFilter()
        elseif g_currentMission.fieldGroundSystem ~= nil and g_currentMission.fieldGroundSystem.getFieldFilter ~= nil then
            self._eraseFoliageCache.fieldFilter = g_currentMission.fieldGroundSystem:getFieldFilter()
        end
    end
end

-- =========================================================
-- ERASE FOLIAGE TO ZERO
-- =========================================================

function FieldGroundMudPhysics:isWorldPosOnFieldByGroundType(worldX, worldZ)
    if g_currentMission == nil or g_currentMission.fieldGroundSystem == nil then
        return true
    end
    if FieldDensityMap == nil or FieldDensityMap.GROUND_TYPE == nil then
        return true
    end

    local value = g_currentMission.fieldGroundSystem:getValueAtWorldPos(FieldDensityMap.GROUND_TYPE, worldX, 0, worldZ)
    if value == nil then
        return true
    end

    local groundType = g_currentMission.fieldGroundSystem:getFieldGroundTypeByValue(value)
    return groundType ~= nil and groundType ~= FieldGroundType.NONE
end

function FieldGroundMudPhysics:eraseFoliageToZero(x0, z0, x1, z1, x2, z2)
    if g_server == nil then return end
    if g_currentMission == nil then return end
    if g_fruitTypeManager == nil or g_fruitTypeManager.getFruitTypes == nil then return end
    if g_terrainNode == nil or g_terrainNode == 0 then return end
	
    if self.eraseFoliageOnlyFields then
        local cx = (x0 + x1 + x2) / 3
        local cz = (z0 + z1 + z2) / 3
        if not self:isWorldPosOnFieldByGroundType(cx, cz) then
            return
        end
    end

    self:initEraseFoliageCache()

    local mm = self._eraseFoliageMM
    local cache = self._eraseFoliageCache
    if mm == nil or cache == nil then return end

    mm:reset()

    local fieldFilter = cache.fieldFilter

    local function keyOf(plane, start, chans)
        return tostring(plane) .. ":" .. tostring(start) .. ":" .. tostring(chans)
    end

    local function getOrCreate(modTable, filtTable, plane, start, chans)
        if plane == nil or plane == 0 then return nil, nil end
        if type(start) ~= "number" or type(chans) ~= "number" or chans <= 0 then return nil, nil end

        local k = keyOf(plane, start, chans)

        local m = modTable[k]
        local f = (filtTable ~= nil) and filtTable[k] or nil

        if m == nil then
            m = DensityMapModifier.new(plane, start, chans, g_terrainNode)
            modTable[k] = m
        end
        if filtTable ~= nil and f == nil then
            f = DensityMapFilter.new(plane, start, chans)
            filtTable[k] = f
        end

        return m, f
    end

	local function addZeroSet(modifier, filter)
		if modifier == nil then return end

		modifier:setParallelogramWorldCoords(x0, z0, x1, z1, x2, z2, DensityCoordType.POINT_POINT_POINT)

		if filter ~= nil then
			filter:setValueCompareParams(DensityValueCompareType.GREATER, 0)
			mm:addExecuteSet(0, modifier, filter)
		else
			mm:addExecuteSet(0, modifier)
		end
	end

    local gs = g_currentMission.growthSystem
    if gs ~= nil and gs.setIgnoreDensityChanges ~= nil then
        gs:setIgnoreDensityChanges(true)
    end

    -- 1) FRUIT -> 0
    for _, ft in ipairs(g_fruitTypeManager:getFruitTypes()) do
        local m, f = getOrCreate(cache.fruitMods, cache.fruitFilts, ft.terrainDataPlaneId, ft.startStateChannel, ft.numStateChannels)
        addZeroSet(m, f)
    end

    -- 2) HAULM -> 0
    for _, ft in ipairs(g_fruitTypeManager:getFruitTypes()) do
        local m, f = getOrCreate(cache.haulmMods, cache.haulmFilts, ft.terrainDataPlaneIdHaulm, ft.startStateChannelHaulm, ft.numStateChannelsHaulm)
        addZeroSet(m, f)
    end

    -- 3) DYNAMIC FOLIAGE LAYERS -> 0 
    if g_currentMission.dynamicFoliageLayers ~= nil then
        for i = 1, #g_currentMission.dynamicFoliageLayers do
            local layerId = g_currentMission.dynamicFoliageLayers[i]
            if layerId ~= nil and layerId ~= 0 then
                local numCh = getTerrainDetailNumChannels(layerId)
                if type(numCh) == "number" and numCh > 0 then
                    local k = "dyn:" .. tostring(layerId) .. ":" .. tostring(numCh)
                    local m = cache.dynMods[k]
                    if m == nil then
                        m = DensityMapModifier.new(layerId, 0, numCh, g_terrainNode)
                        cache.dynMods[k] = m
                    else
                        m:resetDensityMapAndChannels(layerId, 0, numCh)
                    end

                    m:setParallelogramWorldCoords(x0, z0, x1, z1, x2, z2, DensityCoordType.POINT_POINT_POINT)
                    mm:addExecuteSet(0, m)
                end
            end
        end
    end

    mm:execute()

    if gs ~= nil and gs.setIgnoreDensityChanges ~= nil then
        gs:setIgnoreDensityChanges(false)
    end

    -- 4) Weed/Stones отдельными системами (как ваниль)
    if FSDensityMapUtil ~= nil then
        if FSDensityMapUtil.removeWeedArea ~= nil then
            FSDensityMapUtil.removeWeedArea(x0, z0, x1, z1, x2, z2)
        end
        if FSDensityMapUtil.removeStoneArea ~= nil then
            FSDensityMapUtil.removeStoneArea(x0, z0, x1, z1, x2, z2)
        end
    end

    -- 5) Доп. проход “mulcher”)
    if FSDensityMapUtil ~= nil and FSDensityMapUtil.updateMulcherArea ~= nil then
        FSDensityMapUtil.updateMulcherArea(x0, z0, x1, z1, x2, z2, true)
    end
	
	-- 6) УБИВАЕМ "пеньки / cut" (стерню) => ставим stubbleShred в MAX (БЕЗ fieldFilter)
	if FSDensityMapUtil ~= nil and FSDensityMapUtil.setStubbleShredLevelArea ~= nil then
		local fgs = g_currentMission.fieldGroundSystem
		if fgs ~= nil and fgs.getDensityMapData ~= nil then
			local _, _, numCh = fgs:getDensityMapData(FieldDensityMap.STUBBLE_SHRED_LEVEL)
			if type(numCh) == "number" and numCh > 0 then
				local maxVal = (2 ^ numCh) - 1
				FSDensityMapUtil.setStubbleShredLevelArea(x0, z0, x1, z1, x2, z2, maxVal)
			end
		end
	end
end

function FieldGroundMudPhysics:applyRutFromRadius(wp, wx, wz, r0, desiredRadius, minRadius, slip, speedKph)
    if not self.rutsEnable then return end
    if g_currentMission == nil or TerrainDeformation == nil then return end
    if g_currentMission.terrainRootNode == nil then return end

    if speedKph ~= nil and speedKph < (self.rutsSpeedMinKph or 0) then
        return
    end

    local denom = math.max(0.001, (r0 - minRadius))
    local sinkFrac = clamp((r0 - (desiredRadius or r0)) / denom, 0, 1)

    if sinkFrac < (self.rutsMinSinkFrac or 0.05) then
        return
    end

    local nowMs = g_time or 0
    wp.__fgRutLastMs = wp.__fgRutLastMs or 0
    if (nowMs - wp.__fgRutLastMs) < (self.rutsIntervalMs or 150) then
        return
    end
    wp.__fgRutLastMs = nowMs

    local slip01 = clamp((slip or 0) * 1.6, 0, 1) -- нормализуем slip примерно в 0..1
    local bonus  = 1.0 + (self.rutsSlipBonusMul or 0) * slip01

    local depth = (self.rutsMaxDepthM or 0.06) * sinkFrac * bonus
    if depth <= 0.001 then return end

    local w = 0.5
    if wp.wheel ~= nil then
        w = wp.wheel.width or w
    end
    local brushRadius = math.max(0.18, w * (self.rutsRadiusMul or 0.55))

    local deformer = TerrainDeformation.new(g_currentMission.terrainRootNode)
    if deformer == nil then return end

    deformer:enableAdditiveDeformationMode()
    deformer:setAdditiveHeightChangeAmount(-depth)

    local hardness = clamp(self.rutsHardness or 0.55, 0.05, 0.98)
    deformer:addSoftCircleBrush(wx, wz, brushRadius, hardness, 1.0, TerrainDeformation.NO_TERRAIN_BRUSH)

    local q = g_currentMission.terrainDeformationQueue
    if q ~= nil and q.queueJob ~= nil then
        local ok = pcall(function()
            q:queueJob(deformer, false, "onRutDeformDone", self, nil)
        end)

        if not ok then
            ok = pcall(function()
                q:queueJob(deformer, "onRutDeformDone", self, nil, false)
            end)
        end

        if not ok then
            deformer:apply(false, "onRutDeformDone", self, nil)
            deformer:delete()
        end
    else
        deformer:apply(false, "onRutDeformDone", self, nil)
        deformer:delete()
    end

end

function FieldGroundMudPhysics:onRutDeformDone(state, numDeformations, msg)
end

-- =========================================================
-- MP RADIUS SYNC (server -> clients)
-- =========================================================
FieldGroundMudPhysics.mpRadiusSyncIntervalMs = 80

function FieldGroundMudPhysics:collectWheelDesiredRadii(vehicle)
    if vehicle == nil or vehicle.spec_wheels == nil or vehicle.spec_wheels.wheels == nil then
        return nil
    end
    local wheels = vehicle.spec_wheels.wheels
    local radii = {}
    for i = 1, #wheels do
        local w = wheels[i]
        if w ~= nil and w.physics ~= nil then
            local wp = w.physics
            local r = wp.__fgDesiredRadius
            if type(r) == "number" and r == r and r > 0.01 then
                radii[i] = r
            else
                radii[i] = nil
            end
        end
    end
    return radii
end

function FieldGroundMudPhysics:syncWheelRadiiServer(vehicle)
    if g_server == nil then return end
    if self.mpRadiusSyncEnable ~= true then return end
    if vehicle == nil or vehicle.isServer ~= true then return end
    if self:isInShopPreview(vehicle) then return end
    if self:isVehicleResetting(vehicle) then return end

    local now = g_time or 0
    vehicle.__fgLastRadiusSyncMs = vehicle.__fgLastRadiusSyncMs or 0
    if (now - vehicle.__fgLastRadiusSyncMs) < (self.mpRadiusSyncIntervalMs or 80) then
        return
    end
    vehicle.__fgLastRadiusSyncMs = now

    local radii = self:collectWheelDesiredRadii(vehicle)
    if radii == nil then return end

    g_server:broadcastEvent(FieldGroundWheelRadiusSyncEvent.new(vehicle, radii), false, nil, vehicle)
end

-- =========================================================
-- FIELD GROUND TYPE QUERY
-- =========================================================
function FieldGroundMudPhysics:getGroundTypeValueAtWorldPos(x, z)
    if g_currentMission == nil or g_currentMission.fieldGroundSystem == nil then
        return nil
    end

    local fgs = g_currentMission.fieldGroundSystem
    if fgs.getValueAtWorldPos == nil then
        return nil
    end

    local levelType = (FieldDensityMap ~= nil and FieldDensityMap.GROUND_TYPE) or 1
    local v = fgs:getValueAtWorldPos(levelType, x, 0, z)
    return v
end

function FieldGroundMudPhysics:getProfileAtWorldPos(x, z)
    local gt = self:getGroundTypeValueAtWorldPos(x, z)
    if gt == nil then
        return 0, nil, nil
    end

    local prof = self.groundProfiles[gt]
    if prof == nil then
        return 0, gt, nil
    end

    return clamp(prof.mud or 0, 0, 1), gt, prof
end

-- =========================================================
-- VEHICLE STATE
-- =========================================================
FieldGroundMudPhysics.vehicleState = setmetatable({}, { __mode="k" })

getState = function(vehicle)
    local st = FieldGroundMudPhysics.vehicleState[vehicle]
    if st == nil then
        st = {
            sink=0,
            stuck=false,
            permaActive=false,
            permaTargetExtra=0,
            permaCd=0,
            lastTick=-1,
            ramp=0,
            origExtTorqueMult=nil,
            lastAppliedMotorLoad=1,
        }
        FieldGroundMudPhysics.vehicleState[vehicle] = st
    end
    return st
end

-- =========================================================
-- COMPUTE AVERAGE FIELD MUD + SLIP + BEST PROFILE
-- =========================================================
function FieldGroundMudPhysics:computeVehicleFieldMud(vehicle)
    if vehicle.spec_wheels == nil or vehicle.spec_wheels.wheels == nil then
        return 0, 0, nil, nil
    end

    local mudSum, slipSum, count = 0, 0, 0
    local bestMud, bestGT, bestProf = 0, nil, nil

    for _, w in ipairs(vehicle.spec_wheels.wheels) do
        local x, _, z = self:getWheelContactPos(w)
        if x ~= nil then
            local mudBase, gt, prof = self:getProfileAtWorldPos(x, z)
            local slip = 0
            if w.physics ~= nil and w.physics.netInfo ~= nil then
                slip = w.physics.netInfo.slip or 0
            end

            mudSum  = mudSum + mudBase
            slipSum = slipSum + slip
            count   = count + 1

            if mudBase > bestMud and prof ~= nil then
                bestMud = mudBase
                bestGT = gt
                bestProf = prof
            end
        end
    end

    if count == 0 then
        return 0, 0, nil, nil
    end

    return mudSum / count, slipSum / count, bestGT, bestProf
end

-- =========================================================
-- VEHICLE RESET/RELOAD GUARD
-- =========================================================

function FieldGroundMudPhysics:isVehicleResetting(vehicle)
    if vehicle == nil then return true end
    if vehicle.isDeleted == true then return true end
    if vehicle.isDeleting == true then return true end
    if vehicle.isResetInProgress == true then return true end
    if vehicle.__fgResetGuard == true then return true end
    return false
end

function FieldGroundMudPhysics:cleanupVehicleState(vehicle, reason)
    if vehicle == nil then return end
    vehicle.__fgResetGuard = true

    local st = (FieldGroundMudPhysics.vehicleState ~= nil) and FieldGroundMudPhysics.vehicleState[vehicle] or nil
    if st ~= nil then
        st.sink = 0
        st.stuck = false
        st.ramp = 0
        st.lastTick = -1
        st.__fgAppliedSpeedLimit = false
        st.__fgAppliedMotorLoad = false

        st.__fgOrigMotorSpeedLimit = nil
        st.__fgSpeedLimitCaptured = false

        if st.extraPS ~= nil then
            for _, entry in pairs(st.extraPS) do
                if entry ~= nil and entry.ps ~= nil then
                    entry.ps.isActive = false
                    ParticleUtil.setEmittingState(entry.ps, false)
                    if entry.ps.shape ~= nil then
                        mpSafeDelete(entry.ps.shape)
                    end
                end
                if entry ~= nil and entry.shape ~= nil then
                    mpSafeDelete(entry.shape)
                end
            end
            st.extraPS = nil
        end
    end

    local motor = nil
    if vehicle.getMotor ~= nil then motor = vehicle:getMotor() end
    if motor == nil and vehicle.spec_motorized ~= nil then motor = vehicle.spec_motorized.motor end

    if motor ~= nil then
		if motor.setSpeedLimit ~= nil then
			motor:setSpeedLimit(math.huge)
		end


        if motor.setExternalTorqueVirtualMultiplicator ~= nil then
            motor:setExternalTorqueVirtualMultiplicator(1)
        end
    end

    if vehicle.spec_wheels ~= nil and vehicle.spec_wheels.wheels ~= nil then
        for _, w in ipairs(vehicle.spec_wheels.wheels) do
            if w ~= nil and w.physics ~= nil then
                local wp = w.physics
                wp.__fgDesiredRadius = nil
                wp.__fgMudCurExtra = 0
                wp.__fgMudBrakeExtra = 0
                wp.__fgBrakeCur = 0
                wp.__fgPerma = false
                wp.__fgWasMud = false
                wp.__fgStillT = 0
                wp.__fgGripMul = nil
                wp.isFrictionDirty = true

                if _G.__MudRadiusCombiner ~= nil and _G.__MudRadiusCombiner.apply ~= nil then
                    _G.__MudRadiusCombiner.apply(wp, FieldGroundMudPhysics.freezeRadiusEps or 0.0015)
                end
            end
        end
    end

    if FieldGroundMudPhysics.debug then
        print(string.format("[FieldGroundMudPhysics] cleanupVehicleState(%s) reason=%s",
            tostring(vehicle.configFileName or vehicle.typeName or "vehicle"), tostring(reason)))
    end
end

-- =========================================================
-- APPLY SETTINGS IMMEDIATELY
-- =========================================================

function FieldGroundMudPhysics:resetVehicleRuntime(vehicle, reason)
    if vehicle == nil then return end

    local st = (FieldGroundMudPhysics.vehicleState ~= nil) and FieldGroundMudPhysics.vehicleState[vehicle] or nil
    if st ~= nil then
        st.sink = 0
        st.stuck = false
        st.ramp = 0
        st.lastTick = -1

        st.__fgAppliedSpeedLimit = false
        st.__fgAppliedMotorLoad  = false
        st.lastAppliedMotorLoad  = nil

        st.__fgOrigMotorSpeedLimit = nil
        st.__fgSpeedLimitCaptured = false

        if st.extraPS ~= nil then
            for _, entry in pairs(st.extraPS) do
                if entry ~= nil and entry.ps ~= nil then
                    entry.ps.isActive = false
                    ParticleUtil.setEmittingState(entry.ps, false)
                    if entry.ps.shape ~= nil then
                        mpSafeDelete(entry.ps.shape)
                    end
                end
                if entry ~= nil and entry.shape ~= nil then
                    mpSafeDelete(entry.shape)
                end
            end
            st.extraPS = nil
        end
    end

    local motor = nil
    if vehicle.getMotor ~= nil then motor = vehicle:getMotor() end
    if motor == nil and vehicle.spec_motorized ~= nil then motor = vehicle.spec_motorized.motor end

    if motor ~= nil then
        if motor.setExternalTorqueVirtualMultiplicator ~= nil then
            motor:setExternalTorqueVirtualMultiplicator(1)
        end
		if motor.setSpeedLimit ~= nil then
			motor:setSpeedLimit(math.huge)
		end

    end

    if vehicle.spec_wheels ~= nil and vehicle.spec_wheels.wheels ~= nil then
        for _, w in ipairs(vehicle.spec_wheels.wheels) do
            if w ~= nil and w.physics ~= nil then
                local wp = w.physics
                wp.__fgDesiredRadius  = nil
                wp.__fgMudCurExtra    = 0
                wp.__fgMudBrakeExtra  = 0
                wp.__fgBrakeCur       = 0
                wp.__fgPerma          = false
                wp.__fgWasMud         = false
                wp.__fgStillT         = 0
                wp.__fgGripMul        = nil
                wp.isFrictionDirty    = true

                if _G.__MudRadiusCombiner ~= nil and _G.__MudRadiusCombiner.apply ~= nil then
                    _G.__MudRadiusCombiner.apply(wp, FieldGroundMudPhysics.freezeRadiusEps or 0.0015)
                end
            end
        end
    end

    if FieldGroundMudPhysics.debug then
        print(string.format("[FieldGroundMudPhysics] resetVehicleRuntime(%s) reason=%s",
            tostring(vehicle.configFileName or vehicle.typeName or "vehicle"), tostring(reason)))
    end
end

function FieldGroundMudPhysics:onSettingsChanged()
    self.radiusMinFactor    = clamp(tonumber(self.radiusMinFactor) or 0, 0, 1)
    self.wheelBrakeBase     = clamp(tonumber(self.wheelBrakeBase) or 0, 0, 15)
    self.wheelBrakeFromSink = clamp(tonumber(self.wheelBrakeFromSink) or 0, 0, 15)
    self.wheelBrakeFromSlip = clamp(tonumber(self.wheelBrakeFromSlip) or 0, 0, 15)

    if g_currentMission ~= nil and g_currentMission.vehicles ~= nil then
        for _, v in pairs(g_currentMission.vehicles) do
            if v ~= nil then
                self:resetVehicleRuntime(v, "onSettingsChanged")
            end
        end
    end

    if FieldGroundMudPhysics.debug then
        print(string.format("[FieldGroundMudPhysics] settings changed -> enabled=%s", tostring(self.enabled)))
    end
	
    if g_currentMission ~= nil and g_currentMission.vehicles ~= nil then
        for _, v in pairs(g_currentMission.vehicles) do
            if v ~= nil and v.spec_wheels ~= nil and v.spec_wheels.wheels ~= nil then
                for _, w in ipairs(v.spec_wheels.wheels) do
                    if w ~= nil and w.physics ~= nil then
                        w.physics.__fgEraseLastMs = 0
                    end
                end
            end
        end
    end
end

function FieldGroundMudPhysics:onVehicleReset(oldVehicle, newVehicle)
    if self.vehicleState ~= nil and oldVehicle ~= nil then
        local st = self.vehicleState[oldVehicle]
        if st ~= nil and st.extraPS ~= nil then
            for _, entry in pairs(st.extraPS) do
                if entry ~= nil and entry.ps ~= nil then
                    entry.ps.isActive = false
                    ParticleUtil.setEmittingState(entry.ps, false)
                end
            end
        end
        self.vehicleState[oldVehicle] = nil
    end

    if oldVehicle ~= nil and oldVehicle.spec_wheels ~= nil and oldVehicle.spec_wheels.wheels ~= nil then
        for _, w in ipairs(oldVehicle.spec_wheels.wheels) do
            if w ~= nil and w.physics ~= nil then
                local wp = w.physics
                wp.__fgDesiredRadius = nil
                wp.__fgMudCurExtra = 0
                wp.__fgMudBrakeExtra = 0
            end
        end
    end
end

function FieldGroundMudPhysics.installResetHooks()
    if Utils == nil or Utils.overwrittenFunction == nil then
        print("[FieldGroundMudPhysics] Utils not found (installResetHooks)"); return
    end

    if Vehicle ~= nil and Vehicle.reset ~= nil then
        Vehicle.reset = Utils.overwrittenFunction(
            Vehicle.reset,
            function(vehicle, superFunc, forceDelete, callback, resetPlayer)
                if FieldGroundMudPhysics ~= nil then
                    FieldGroundMudPhysics:cleanupVehicleState(vehicle, "Vehicle.reset")
                end
                if superFunc ~= nil then
                    return superFunc(vehicle, forceDelete, callback, resetPlayer)
                end
            end
        )
    end

    if Vehicle ~= nil and Vehicle.delete ~= nil then
        Vehicle.delete = Utils.overwrittenFunction(
            Vehicle.delete,
            function(vehicle, superFunc, ...)
                if FieldGroundMudPhysics ~= nil then
                    FieldGroundMudPhysics:cleanupVehicleState(vehicle, "Vehicle.delete")
                end
                if superFunc ~= nil then
                    return superFunc(vehicle, ...)
                end
            end
        )
    end

    if g_messageCenter ~= nil and MessageType ~= nil and MessageType.VEHICLE_RESET ~= nil then
        g_messageCenter:subscribe(MessageType.VEHICLE_RESET, FieldGroundMudPhysics.onVehicleReset, FieldGroundMudPhysics)
    end

    print("[FieldGroundMudPhysics] reset hooks installed (Vehicle.reset/delete + MessageType.VEHICLE_RESET)")
end

function FieldGroundMudPhysics.deleteResetHooks()
    if g_messageCenter ~= nil and MessageType ~= nil and MessageType.VEHICLE_RESET ~= nil then
        g_messageCenter:unsubscribe(MessageType.VEHICLE_RESET, FieldGroundMudPhysics)
    end
end

-- =========================================================
-- DIRT / MUD BUILDUP (Washable) - FIELD GROUND TYPES
-- =========================================================
function FieldGroundMudPhysics:applyFieldDirt(vehicle, dt, wetnessEff, speedKph, rampPowVal)
    if not self.dirtEnable then
        return
    end

    if self.isInShopPreview ~= nil and self:isInShopPreview(vehicle) then
        return
    end

	if vehicle == nil then
		return
	end

    if vehicle.getDirtAmount == nil or vehicle.setDirtAmount == nil then
        return
    end

    local dtS = (dt or 0) * 0.001
    if dtS <= 0 then
        return
    end

    if (wetnessEff or 0) < (self.dirtWetnessMin or 0.18) then
        return
    end

    local sKph = speedKph or 0
    local spDiv = math.max(1.0, self.dirtSpeedBoostKph or 18.0)
    local speedFactor = 1.0 + math.min(sKph / spDiv, 1.0)

    local dirtMax = self.dirtMax or 1.0
    local rampK = rampPowVal or 1.0

    local mudAvg, _, bestGT, bestProf = self:computeVehicleFieldMud(vehicle)

    local wetFactor = clamp(((wetnessEff or 0) - 0.15) / 0.55, 0, 1)
    local wetMul = (bestProf ~= nil and bestProf.wetMul) or 1.0
    local effMudBody = clamp((mudAvg or 0) * wetFactor * wetMul, 0, 1) * rampK

    local profDirtMulBody = (bestProf ~= nil and bestProf.dirtMul) or 1.0

    if effMudBody >= (self.dirtMinEffMud or 0.06) then
        local bodyAdd = (self.dirtBodyPerSec or 0.020) * effMudBody * profDirtMulBody * speedFactor * dtS
        if bodyAdd > 0 then
            local cur = vehicle:getDirtAmount() or 0
            vehicle:setDirtAmount(clamp(cur + bodyAdd, 0, dirtMax))
        end
    end

    if vehicle.spec_wheels == nil or vehicle.spec_wheels.wheels == nil then
        return
    end

    if vehicle.getWashableNodeByCustomIndex == nil or vehicle.setNodeDirtAmount == nil then
        return
    end

    local wheels = vehicle.spec_wheels.wheels
    local minMud = (self.dirtMinEffMud or 0.06)

    for _, w in ipairs(wheels) do
        if w ~= nil then
            local x, _, z = self:getWheelContactPos(w)
            if x ~= nil then
                local mudBase, gt, prof = self:getProfileAtWorldPos(x, z)

                if prof ~= nil and (mudBase or 0) > 0.01 then
                    local wetMulW = prof.wetMul or 1.0
                    local effMudWheel = clamp((mudBase or 0) * wetFactor * wetMulW, 0, 1) * rampK

                    effMudWheel = math.max(effMudWheel, (mudBase or 0) * 0.20 * rampK)

                    if effMudWheel >= minMud then
                        local profDirtMul = prof.dirtMul or 1.0
                        local wheelAdd = (self.dirtWheelPerSec or 0.085) * effMudWheel * profDirtMul * speedFactor * dtS

                        if wheelAdd > 0 then
                            local nd = vehicle:getWashableNodeByCustomIndex(w)
                            if nd ~= nil and nd.dirtAmount ~= nil then
                                vehicle:setNodeDirtAmount(nd, clamp((nd.dirtAmount or 0) + wheelAdd, 0, dirtMax), true)
                                w.forceWheelDirtUpdate = true
                            end

                            if w.wheelMudMeshes ~= nil then
                                local ndMud = vehicle:getWashableNodeByCustomIndex(w.wheelMudMeshes)
                                if ndMud ~= nil and ndMud.dirtAmount ~= nil then
                                    vehicle:setNodeDirtAmount(ndMud, clamp((ndMud.dirtAmount or 0) + wheelAdd, 0, dirtMax), true)
                                    w.forceWheelDirtUpdate = true
                                end
                            end
                        end
                    end
                end
            end
        end
    end
end

-- =========================================================
-- MOTOR LOAD
-- =========================================================
function FieldGroundMudPhysics:applyMotorLoad(vehicle, effMud, slip, prof)
    if not self.motorLoadEnable then return end

    local spec = vehicle.spec_motorized
    if spec == nil or spec.motor == nil or spec.motor.setExternalTorqueVirtualMultiplicator == nil then
        return
    end

    local st = getState(vehicle)
    st.__fgAppliedMotorLoad = st.__fgAppliedMotorLoad or false

    if st.origExtTorqueMult == nil then
        st.origExtTorqueMult = 1
    end

    local mulProf = (prof ~= nil and prof.motorMul) or 1.0

    if effMud <= (self.motorLoadMinEff or 0) then
        if st.__fgAppliedMotorLoad then
            spec.motor:setExternalTorqueVirtualMultiplicator(st.origExtTorqueMult)
            st.lastAppliedMotorLoad = st.origExtTorqueMult
            st.__fgAppliedMotorLoad = false
        end
        return
    end

    local mult = 1
        + (effMud * mulProf) * (st.sink * (self.motorLoadFromSink or 0))
        + (effMud * mulProf) * (slip * (self.motorLoadFromSlip or 0))

    if st.stuck then
        mult = self.motorLoadMaxMult or mult
    end

    mult = clamp(mult, 1, self.motorLoadMaxMult or 5)

    if math.abs((st.lastAppliedMotorLoad or 1) - mult) > 0.01 then
        spec.motor:setExternalTorqueVirtualMultiplicator(mult)
        st.lastAppliedMotorLoad = mult
    end

    st.__fgAppliedMotorLoad = true
end

-- =========================================================
-- EXTRA PARTICLES
-- =========================================================
function FieldGroundMudPhysics:loadExtraParticleReferences()
    if not self.extraParticlesEnable then return end
    self._extraPS = self._extraPS or {}

    if self._extraPS.referencePS ~= nil and self._extraPS.referenceShape ~= nil then
        return
    end

    local psFile = Utils.getFilename(self.extraParticlesI3D, self.path)
    local root = loadI3DFile(psFile, false, false, false)
    if root == nil or root == 0 then
        print(string.format("[FieldGroundMudPhysics] ExtraPS: failed to load '%s'", tostring(psFile)))
        return
    end

    local psNode = I3DUtil.indexToObject(root, "0")
    if psNode == nil then
        print(string.format("[FieldGroundMudPhysics] ExtraPS: node '0' not found in '%s'", tostring(psFile)))
        mpSafeDelete(root)
        return
    end

    local ps = {}
    ParticleUtil.loadParticleSystemFromNode(psNode, ps, false, true, false)
    link(getRootNode(), ps.shape)
    setClipDistance(ps.shape, self.extraParticleClipDist or 90)
    if ps.geometry ~= nil then
        setMaxNumOfParticles(ps.geometry, self.extraParticleMaxCount or 900)
    end
    self._extraPS.referencePS = ps

    local shapePath = Utils.getFilename(self.extraEmitterShapeI3D, self.path)
    local shapeRoot = loadSharedI3DFile(shapePath, false, false)
    if shapeRoot == nil or shapeRoot == 0 then
        print(string.format("[FieldGroundMudPhysics] ExtraPS: failed to load emitterShape '%s'", tostring(shapePath)))
        return
    end

    local refShape = getChildAt(shapeRoot, 0)
    if refShape == nil then
        print(string.format("[FieldGroundMudPhysics] ExtraPS: emitterShape child 0 missing '%s'", tostring(shapePath)))
        mpSafeDelete(shapeRoot)
        return
    end

    link(getRootNode(), refShape)
    setClipDistance(refShape, self.extraParticleClipDist or 90)
    self._extraPS.referenceShape = refShape
end

local function fgGetRootNodeForWheel(vehicle, wheel)
    if vehicle == nil then
        return nil
    end

    if wheel ~= nil and wheel.node ~= nil and vehicle.vehicleNodes ~= nil then
        local vNode = vehicle.vehicleNodes[wheel.node]
        if vNode ~= nil and vNode.component ~= nil and vNode.component.node ~= nil and vNode.component.node ~= 0 then
            return vNode.component.node
        end
    end

    if vehicle.getRootNode ~= nil then
        local rn = vehicle:getRootNode()
        if rn ~= nil and rn ~= 0 then
            return rn
        end
    end

    if vehicle.components ~= nil and vehicle.components[1] ~= nil then
        local n = vehicle.components[1].node
        if n ~= nil and n ~= 0 then
            return n
        end
    end

    return vehicle.rootNode
end

local function fgWheelKey(wheel)
    if wheel ~= nil then
        return wheel
    end
    return "nilWheel"
end

local function ensureWheelExtraPS(vehicle, wheelRef)
    FieldGroundMudPhysics.vehicleState = FieldGroundMudPhysics.vehicleState or setmetatable({}, { __mode="k" })
    local st = FieldGroundMudPhysics.vehicleState[vehicle]
    if st == nil then
        st = { extraPS = {} }
        FieldGroundMudPhysics.vehicleState[vehicle] = st
    end
    st.extraPS = st.extraPS or {}

    local key = fgWheelKey(wheelRef)
    local entry = st.extraPS[key]
    if entry ~= nil and entry.ps ~= nil and entry.emitter ~= nil then
        return entry
    end

    if FieldGroundMudPhysics._extraPS == nil
        or FieldGroundMudPhysics._extraPS.referencePS == nil
        or FieldGroundMudPhysics._extraPS.referenceShape == nil then
        return nil
    end

    local rootNode = fgGetRootNodeForWheel(vehicle, wheelRef)
    if rootNode == nil or rootNode == 0 then
        return nil
    end

    local emitter = createTransformGroup("fgExtraMudEmitter")
    link(rootNode, emitter)

    local emitterShape = clone(FieldGroundMudPhysics._extraPS.referenceShape, true, false, true)
    link(emitter, emitterShape)

    local psClone = clone(FieldGroundMudPhysics._extraPS.referencePS.shape, true, false, true)
    link(emitter, psClone)

    local psTbl = {}
    ParticleUtil.loadParticleSystemFromNode(psClone, psTbl, false, true, false)
    ParticleUtil.setEmitterShape(psTbl, emitterShape)
    ParticleUtil.setEmittingState(psTbl, false)
    psTbl.isActive = false

    entry = { emitter=emitter, ps=psTbl, __fgRootNode=rootNode }
    st.extraPS[key] = entry
    return entry
end

local function updateWheelExtraPS(vehicle, wheelRef, isActive, wx, wy, wz)
    local entry = ensureWheelExtraPS(vehicle, wheelRef)
    if entry == nil then return end

    local rootNode = fgGetRootNodeForWheel(vehicle, wheelRef)
    if rootNode ~= nil and rootNode ~= 0 then
        if entry.__fgRootNode ~= rootNode then
            link(rootNode, entry.emitter)
            entry.__fgRootNode = rootNode
        end

        if wx ~= nil then
            local lx, ly, lz = worldToLocal(rootNode, wx, wy, wz)
            ly = ly + (FieldGroundMudPhysics.extraParticleOffsetY or -0.24)
            setTranslation(entry.emitter, lx, ly, lz)
        end
    end

    local ps = entry.ps
    if ps ~= nil then
        if ps.isActive ~= isActive then
            ps.isActive = isActive
            ParticleUtil.setEmittingState(ps, isActive)
        end
    end
end

-- =========================================================
-- PHYSICS APPLY
-- =========================================================
function FieldGroundMudPhysics:applyFieldPhysics(vehicle, dt, mudBaseAvg, slipAvg, wetnessEff, bestGT, bestProf)
    local st = getState(vehicle)

    if self:isInShopPreview(vehicle) then
        if vehicle.spec_motorized ~= nil and vehicle.spec_motorized.motor ~= nil
            and vehicle.spec_motorized.motor.setExternalTorqueVirtualMultiplicator ~= nil then
            vehicle.spec_motorized.motor:setExternalTorqueVirtualMultiplicator(1)
        end
        st.sink = 0
        st.stuck = false
        st.ramp = 0
        return
    end

    local dtS = (dt or 0) * 0.001
    if dtS <= 0 then return end

    local wetFactor = clamp((wetnessEff - 0.15) / 0.55, 0, 1)
    local wetMul = (bestProf ~= nil and bestProf.wetMul) or 1.0
    local effMud = clamp((mudBaseAvg or 0) * wetFactor * wetMul, 0, 1)

    local inSec  = math.max(0.05, self.rampInSec or 8.0)
    local outSec = math.max(0.05, self.rampOutSec or 3.0)
    local powK   = math.max(0.25, self.rampPow or 1.5)

    local inMud = effMud > 0.05
    if inMud then
        st.ramp = math.min(1, (st.ramp or 0) + dtS / inSec)
    else
        st.ramp = math.max(0, (st.ramp or 0) - dtS / outSec)
    end

    local ramp = (st.ramp or 0) ^ powK
    local effMudRamp = effMud * ramp

    local vSpeed = (vehicle.getLastSpeed and vehicle:getLastSpeed()) or vehicle.lastSpeedReal or 0
    local speedKph = mpsToKph(vSpeed)

    if self.applyFieldDirt ~= nil then
        self:applyFieldDirt(vehicle, dt, wetnessEff, speedKph, ramp)
    end

    local sinkMul = (bestProf ~= nil and bestProf.sinkMul) or 1.0
    local sinkIn  = (effMudRamp * sinkMul) * (0.30 + (slipAvg or 0)) * (self.sinkInSpeed or 1.0)
    local sinkOut = (1 - effMudRamp) * (speedKph / 25) * (self.sinkOutSpeed or 0.35)

    st.sink = clamp((st.sink or 0) + dtS * (sinkIn - sinkOut), 0, 1)
    st.stuck = (st.sink or 0) >= (self.stuckThreshold or 0.82)

    local motor = nil
    if vehicle.getMotor ~= nil then motor = vehicle:getMotor() end
    if motor == nil and vehicle.spec_motorized ~= nil then motor = vehicle.spec_motorized.motor end

    st.__fgAppliedSpeedLimit = st.__fgAppliedSpeedLimit or false
    st.__fgSpeedLimitCaptured = st.__fgSpeedLimitCaptured or false

    if motor ~= nil and motor.setSpeedLimit ~= nil then
        local isReallyWet = (wetnessEff or 0) >= (self.wetnessThreshold or 0.12)
        local wantMudLogic = isReallyWet and (effMudRamp > 0.05)

        local profMudLimitKph = (bestProf ~= nil) and bestProf.maxSpeedMudKph or nil

        if wantMudLogic and not st.__fgSpeedLimitCaptured then
            local orig = motor.speedLimit
			if type(orig) ~= "number" then
				orig = math.huge
			end
			st.__fgOrigMotorSpeedLimit = orig

            st.__fgSpeedLimitCaptured = true
        end

        if wantMudLogic then
            local wantLimitKph = nil

            if st.stuck then
                wantLimitKph = (self.maxSpeedStuckKph or 3.5)

            elseif type(profMudLimitKph) == "number" and profMudLimitKph > 0 then
                wantLimitKph = profMudLimitKph * (1 - (st.sink or 0) * 0.75)

            else
                local base = self.maxSpeedMudKph or 14.0
                wantLimitKph = base * (1 - (st.sink or 0) * 0.60)
            end

            if type(wantLimitKph) == "number" and wantLimitKph > 0 then
                local wantLimitMps = kphToMps(wantLimitKph)

                local curLimitMps = motor.speedLimit
                if type(curLimitMps) ~= "number" then curLimitMps = math.huge end

                if wantLimitMps < (curLimitMps - 0.05) then
                    motor:setSpeedLimit(wantLimitMps)
                end

                st.__fgAppliedSpeedLimit = true
            else
                if st.__fgAppliedSpeedLimit then
                    motor:setSpeedLimit((type(st.__fgOrigMotorSpeedLimit) == "number") and st.__fgOrigMotorSpeedLimit or math.huge)

                    st.__fgAppliedSpeedLimit = false
                    st.__fgSpeedLimitCaptured = false
                    st.__fgOrigMotorSpeedLimit = nil
                end
            end
        else
            if st.__fgAppliedSpeedLimit then
                motor:setSpeedLimit((type(st.__fgOrigMotorSpeedLimit) == "number") and st.__fgOrigMotorSpeedLimit or math.huge)

                st.__fgAppliedSpeedLimit = false
            end
            st.__fgSpeedLimitCaptured = false
            st.__fgOrigMotorSpeedLimit = nil
        end
    end

    self:applyMotorLoad(vehicle, effMudRamp, slipAvg or 0, bestProf)

    if self.debug then
        local gtStr = (bestGT ~= nil) and tostring(bestGT) or "nil"
        local pn = (bestProf ~= nil and bestProf.name) or "none"
        print(string.format("[FieldGroundMudPhysics] gt=%s(%s) wet=%.2f mud=%.2f eff=%.2f ramp=%.2f sink=%.2f slip=%.2f stuck=%s",
            gtStr, pn, wetnessEff or 0, mudBaseAvg or 0, effMudRamp or 0, ramp, st.sink or 0, slipAvg or 0, tostring(st.stuck)))
    end
end

-- =========================================================
-- HOOKS
-- =========================================================
function FieldGroundMudPhysics.installHooks()
    if Utils == nil or Utils.overwrittenFunction == nil then
        print("[FieldGroundMudPhysics] Utils not found"); return
    end

    if WheelEffects ~= nil and WheelEffects.updateTick ~= nil then
    WheelEffects.updateTick = Utils.overwrittenFunction(
        WheelEffects.updateTick,
        function(self, superFunc, dt, groundWetness, currentUpdateDistance)
			if self ~= nil and self.vehicle ~= nil and FieldGroundMudPhysics:isVehicleResetting(self.vehicle) then
				if superFunc ~= nil then superFunc(self, dt, groundWetness, currentUpdateDistance) end
				return
			end
			
            if not FieldGroundMudPhysics.enabled or self == nil or self.vehicle == nil or self.wheel == nil then
                if superFunc ~= nil then superFunc(self, dt, groundWetness, currentUpdateDistance) end
                return
            end

            -- shop preview: vanilla only
            if FieldGroundMudPhysics:isInShopPreview(self.vehicle) then
                if superFunc ~= nil then superFunc(self, dt, groundWetness, currentUpdateDistance) end
                return
            end

			local wetnessEff = computeWetnessEffective(groundWetness)
			
			do
				if g_client ~= nil then
					local stV = getState(self.vehicle)
					local timeMs = g_time or 0
					if stV.__fgVehTick ~= timeMs then
						stV.__fgVehTick = timeMs

						local mudAvg, slipAvg, bestGT2, bestProf2 = FieldGroundMudPhysics:computeVehicleFieldMud(self.vehicle)
						FieldGroundMudPhysics:applyFieldPhysics(self.vehicle, dt, mudAvg, slipAvg, wetnessEff, bestGT2, bestProf2)
					end
				end
			end
			
			local wetnessFinal = math.max(groundWetness or 0, wetnessEff or 0)

			if superFunc ~= nil then
				superFunc(self, dt, wetnessFinal, currentUpdateDistance)
			end

            -- extraPS только когда колесо реально касается земли
            local hasRealContact = (self.wheel.physics ~= nil and self.wheel.physics.hasSoilContact == true)

            local x, y, z = FieldGroundMudPhysics:getWheelContactPos(self.wheel)
            local mudBase, gt, prof = 0, nil, nil
            if x ~= nil then
                mudBase, gt, prof = FieldGroundMudPhysics:getProfileAtWorldPos(x, z)
            end

            local wetMud = hasRealContact
                and (prof ~= nil)
                and ((mudBase or 0) > 0.01)
                and (wetnessEff > (FieldGroundMudPhysics.wetnessThreshold or 0.12))

            -- extra particles (fxExtra=true)
            if FieldGroundMudPhysics.extraParticlesEnable and prof ~= nil and prof.fxExtra == true then
                if FieldGroundMudPhysics._extraPS == nil
                    or FieldGroundMudPhysics._extraPS.referencePS == nil
                    or FieldGroundMudPhysics._extraPS.referenceShape == nil then
                    FieldGroundMudPhysics:loadExtraParticleReferences()
                end

                local active = wetMud

                if active then
                    local vSpeed = (self.vehicle.getLastSpeed and self.vehicle:getLastSpeed()) or self.vehicle.lastSpeedReal or 0
                    local speedKph = mpsToKph(vSpeed)
                    if speedKph < (FieldGroundMudPhysics.extraParticleMinSpeedKph or 18.0) then
                        active = false
                    end
                end

                if x ~= nil and updateWheelExtraPS ~= nil then
                    updateWheelExtraPS(self.vehicle, self.wheel, active, x, y or 0, z)
                end
            else
                local st = FieldGroundMudPhysics.vehicleState ~= nil and FieldGroundMudPhysics.vehicleState[self.vehicle] or nil
                if st ~= nil and st.extraPS ~= nil then
                    local entry = st.extraPS[self.wheel]
                    if entry ~= nil and entry.ps ~= nil then
                        entry.ps.isActive = false
                        ParticleUtil.setEmittingState(entry.ps, false)
                    end
                end
            end
        end
    )
    else
        print("[FieldGroundMudPhysics] WheelEffects not found (dedicated/server) - skipping particle hooks")
    end

	if WheelPhysics ~= nil and WheelPhysics.updateTireFriction ~= nil then
		WheelPhysics.updateTireFriction = Utils.overwrittenFunction(
			WheelPhysics.updateTireFriction,
			function(self, superFunc)

				if self ~= nil and self.vehicle ~= nil and FieldGroundMudPhysics:isVehicleResetting(self.vehicle) then
					self.__fgGripMul = nil
					if superFunc ~= nil then superFunc(self) end
					return
				end

				if self ~= nil and self.vehicle ~= nil and FieldGroundMudPhysics:isInShopPreview(self.vehicle) then
					self.__fgGripMul = nil
					if superFunc ~= nil then superFunc(self) end
					return
				end

				local hadScale = (self ~= nil and self.frictionScale ~= nil)
				local oldScale = self.frictionScale

				if self ~= nil and self.vehicle ~= nil and self.vehicle.isAddedToPhysics then
					local mul = self.__fgGripMul
					if type(mul) ~= "number" then mul = 1.0 end
					mul = clamp(mul, 0.05, 5.0)

					local base = oldScale
					if type(base) ~= "number" then base = 1.0 end

					self.frictionScale = base * mul
				end

				if superFunc ~= nil then
					superFunc(self)
				end

				if hadScale then
					self.frictionScale = oldScale
				else
					self.frictionScale = nil
				end
			end
		)
	end

    if WheelPhysics ~= nil and WheelPhysics.updatePhysics ~= nil then
        WheelPhysics.updatePhysics = Utils.overwrittenFunction(
            WheelPhysics.updatePhysics,
            function(self, superFunc, brakeForce, torque)
				if self ~= nil and self.vehicle ~= nil and FieldGroundMudPhysics:isVehicleResetting(self.vehicle) then
					self.__fgMudBrakeExtra = 0
					if superFunc ~= nil then superFunc(self, brakeForce, torque) end
					return
				end

                if self ~= nil and self.vehicle ~= nil and FieldGroundMudPhysics:isInShopPreview(self.vehicle) then
                    if superFunc ~= nil then superFunc(self, brakeForce, torque) end
                    return
                end

                local extra = 0
                if FieldGroundMudPhysics.enabled and FieldGroundMudPhysics.wheelBrakeEnable and self ~= nil then
                    extra = self.__fgMudBrakeExtra or 0
                end

                if superFunc ~= nil then
                    superFunc(self, (brakeForce or 0) + extra, torque)
                end
            end
        )
    end

	if WheelPhysics ~= nil and WheelPhysics.serverUpdate ~= nil then
		WheelPhysics.serverUpdate = Utils.overwrittenFunction(
			WheelPhysics.serverUpdate,
			function(self, superFunc, dt, currentUpdateIndex, groundWetness)

				if superFunc ~= nil then
					superFunc(self, dt, currentUpdateIndex, groundWetness)
				end

				if self == nil or self.vehicle == nil or self.wheel == nil then return end
				if g_server == nil then return end
				if not FieldGroundMudPhysics.enabled then

					self.__fgDesiredRadius = nil
					self.__fgMudCurExtra = 0
					self.__fgMudBrakeExtra = 0
					self.__fgBrakeCur = 0
					self.__fgGripMul = nil
					self.isFrictionDirty = true
					return
				end

				if FieldGroundMudPhysics:isInShopPreview(self.vehicle) or FieldGroundMudPhysics:isVehicleResetting(self.vehicle) then
					self.__fgDesiredRadius = nil
					self.__fgMudCurExtra = 0
					self.__fgMudBrakeExtra = 0
					self.__fgBrakeCur = 0
					self.__fgGripMul = nil
					self.isFrictionDirty = true
					return
				end

				local dtS = (dt or 0) * 0.001
				if dtS <= 0 then return end

				do
					local stV = getState(self.vehicle)
					local timeMs = g_time or 0
					if stV.__fgVehTickS ~= timeMs then
						stV.__fgVehTickS = timeMs
						local wetnessEff = computeWetnessEffective(groundWetness)
						local mudAvg, slipAvg, bestGT2, bestProf2 = FieldGroundMudPhysics:computeVehicleFieldMud(self.vehicle)
						FieldGroundMudPhysics:applyFieldPhysics(self.vehicle, dt, mudAvg, slipAvg, wetnessEff, bestGT2, bestProf2)
					end
				end

				local eps = FieldGroundMudPhysics.freezeRadiusEps or 0.0015

				do
					local ro = self.radiusOriginal

					if (type(ro) ~= "number" or ro ~= ro or ro <= 0.05) and self.wheel ~= nil and self.wheel.physics ~= nil then
						ro = self.wheel.physics.radiusOriginal or self.wheel.physics.radius
					end

					if type(ro) == "number" and ro == ro and ro > 0.05 then
						if self.__fgOrigRadius == nil or ro > self.__fgOrigRadius then
							self.__fgOrigRadius = ro
						end
					else
						local cur = self.radius
						if type(cur) == "number" and cur == cur and cur > 0.05 then
							if self.__fgOrigRadius == nil or cur > self.__fgOrigRadius then
								self.__fgOrigRadius = cur
							end
						end
					end
				end

				local r0 = self.__fgOrigRadius
				if type(r0) ~= "number" or r0 ~= r0 or r0 <= 0.05 then return end

				local wx, _, wz = FieldGroundMudPhysics:getWheelContactPos(self.wheel)
				if wx == nil then
					-- НЕ обнуляем резко: делаем плавное восстановление по outSpeed
					local dtS = (dt or 0) * 0.001
					local cur = self.__fgMudCurExtra or 0

					if cur > 0 and dtS > 0 then
						local outSpd =
							FieldGroundMudPhysics.radiusSinkOutSpeed
							or FieldGroundMudPhysics.radiusRecoverOutOfMudSpeed
							or FieldGroundMudPhysics.sinkOutSpeed
							or 0.85


						cur = math.max(0, cur - outSpd * dtS)
						self.__fgMudCurExtra = cur

						if cur > 0 then
							self.__fgDesiredRadius = clamp(r0 - cur, 0.12, r0)
						else
							self.__fgDesiredRadius = nil
						end

						if _G.__MudRadiusCombiner ~= nil and _G.__MudRadiusCombiner.apply ~= nil then
							_G.__MudRadiusCombiner.apply(self, eps)
						end
					else
						self.__fgDesiredRadius = nil
					end
					return
				end


				local wetnessEff = computeWetnessEffective(groundWetness)
				local mudBase, gt, prof = FieldGroundMudPhysics:getProfileAtWorldPos(wx, wz)
				mudBase = mudBase or 0

				local onMudProfile = (prof ~= nil)
					and (mudBase > 0.01)
					and ((wetnessEff or 0) >= (FieldGroundMudPhysics.wetnessThreshold or 0.12))

				local slip = 0
				if self.netInfo ~= nil then
					slip = self.netInfo.slip or 0
				end
				local vSpeed = (self.vehicle.getLastSpeed and self.vehicle:getLastSpeed()) or self.vehicle.lastSpeedReal or 0
				local speedKph = mpsToKph(vSpeed)

				do
					local gripMul = nil
					if onMudProfile and prof ~= nil and prof.slip ~= nil then
						local mn = FieldGroundMudPhysics.slipMinMul or 0.45
						local mx = FieldGroundMudPhysics.slipMaxMul or 0.95
						gripMul = clamp(prof.slip, mn, mx)
					end
					if self.__fgGripMul ~= gripMul then
						self.__fgGripMul = gripMul
						self.isFrictionDirty = true
					end
				end

				if not FieldGroundMudPhysics.radiusSinkEnable then
					self.__fgDesiredRadius = nil
					self.__fgMudCurExtra = 0
					if _G.__MudRadiusCombiner ~= nil and _G.__MudRadiusCombiner.apply ~= nil then
						_G.__MudRadiusCombiner.apply(self, eps)
					end
					return
				end

				local minFactor = FieldGroundMudPhysics.radiusMinFactor or 0.58
				if onMudProfile and prof ~= nil and prof.radiusMinFactor ~= nil then
					minFactor = prof.radiusMinFactor
				end
				minFactor = clamp(minFactor, 0.12, 0.98)

				local minRadius = math.max(0.12, r0 * minFactor)
				local maxExtra  = math.max(0, r0 - minRadius)

				local effMud = 0
				if onMudProfile and prof ~= nil then
					local wetFactor = clamp((wetnessEff - 0.15) / 0.55, 0, 1)
					local wetMul = (prof.wetMul or 1.0)
					effMud = clamp(mudBase * wetFactor * wetMul, 0, 1)
				end
				
				do
					if FieldGroundMudPhysics.eraseFruitEnable
						and (wetnessEff or 0) >= (FieldGroundMudPhysics.wetnessThreshold or 0.12)
						and (speedKph or 0) >= (FieldGroundMudPhysics.eraseFruitSpeedMinKph or 1.0) then


						-- только если есть РЕАЛЬНЫЙ контакт
						local hasRealContact =
							(self.hasSoilContact == true)
							or (self.hasGroundContact == true)
							or (self.wheel ~= nil and self.wheel.physics ~= nil and self.wheel.physics.hasSoilContact == true)
							or (self.wheel ~= nil and self.wheel.physics ~= nil and self.wheel.physics.hasGroundContact == true)

						if hasRealContact then
							local nowMs = g_time or 0
							self.__fgEraseLastMs = self.__fgEraseLastMs or 0

							if (nowMs - self.__fgEraseLastMs) >= (FieldGroundMudPhysics.eraseFruitIntervalMs or 220) then
								self.__fgEraseLastMs = nowMs

								local x0, z0, x1, z1, x2, z2 = FieldGroundMudPhysics:getWheelDestructionParallelogram(self.wheel)
								if x0 ~= nil then
									-- 1) ваниль crushed
									if FieldGroundMudPhysics.eraseFruitDoVanillaWheelDestructionFirst
										and FSDensityMapUtil ~= nil
										and FSDensityMapUtil.updateWheelDestructionArea ~= nil then
										FSDensityMapUtil.updateWheelDestructionArea(x0, z0, x1, z1, x2, z2)
									end

									-- 2) жёсткий ноль (полное исчезновение урожая)
									FieldGroundMudPhysics:eraseFoliageToZero(x0, z0, x1, z1, x2, z2)
								end
							end
						end
					end
				end

				local targetExtraMove = 0
				if onMudProfile and prof ~= nil then
					local stV = getState(self.vehicle)
					local sinkMul = prof.sinkMul or 1.0

					local moving01   = clamp(speedKph / 6.0, 0, 1)
					local slip01     = clamp((slip or 0) / 0.18, 0, 1)
					local moveFactor = math.max(moving01, slip01)

					local k = clamp(effMud * sinkMul * (0.10 + moveFactor * (0.85 + (stV.sink or 0) * 0.55)), 0, 1)
					targetExtraMove = (r0 * 0.28) * k
					targetExtraMove = clamp(targetExtraMove, 0, maxExtra)
				end
				
				-- =========================================================
                -- STOP RECOVER LIMIT (anti "pushing out" when standing still)
                -- =========================================================
                local stopped = false
                do
                    if FieldGroundMudPhysics.freezeRadiusWhenStopped then
                        local spdStop = FieldGroundMudPhysics.freezeStopSpeedKph or 0.60
                        local slipStop = FieldGroundMudPhysics.freezeStopSlip or 0.010
                        if (speedKph or 0) <= spdStop and (slip or 0) <= slipStop then
                            stopped = true
                        end
                    end
                end

                if stopped and onMudProfile then
                    self.__fgStillT = (self.__fgStillT or 0) + dtS
                    if self.__fgStopAnchorExtra == nil then
                        self.__fgStopAnchorExtra = self.__fgMudCurExtra or 0
                    end
                else
                    self.__fgStillT = 0
                    self.__fgStopAnchorExtra = nil
                end

                if stopped and onMudProfile and self.__fgStopAnchorExtra ~= nil then
                    local maxRec = FieldGroundMudPhysics.stopRecoverRadiusMaxM or 0.002
                    if maxRec > 0.002 then maxRec = 0.002 end
                    if maxRec < 0 then maxRec = 0 end

                    local minExtra = math.max(0, (self.__fgStopAnchorExtra or 0) - maxRec)

                    if targetExtraMove < minExtra then
                        targetExtraMove = minExtra
                    end
                end

				
				local cur = self.__fgMudCurExtra or 0
				local inSpd  = FieldGroundMudPhysics.radiusSinkInSpeed or 0.095
				local outSpd =
					(onMudProfile and (FieldGroundMudPhysics.radiusRecoverInMudSpeed or 0.02))
					or (FieldGroundMudPhysics.radiusSinkOutSpeed
						or FieldGroundMudPhysics.radiusRecoverOutOfMudSpeed
						or FieldGroundMudPhysics.sinkOutSpeed
						or 0.85)



				if targetExtraMove > cur then
					cur = math.min(targetExtraMove, cur + inSpd * dtS)
				else
					cur = math.max(targetExtraMove, cur - outSpd * dtS)
				end

				self.__fgMudCurExtra = cur

				if cur <= 0.00001 then
					self.__fgDesiredRadius = nil
				else
					self.__fgDesiredRadius = clamp(r0 - cur, minRadius, r0)
				end

				if _G.__MudRadiusCombiner ~= nil and _G.__MudRadiusCombiner.apply ~= nil then
					_G.__MudRadiusCombiner.apply(self, eps)
				end

			end
		)
	end

print("[FieldGroundMudPhysics] hooks installed (WheelEffects + WheelPhysics)")
end

-- =========================================================
-- MOD EVENTS
-- =========================================================
function FieldGroundMudPhysics:loadMap()
    self:loadExtraParticleReferences()
    self.installHooks()
	FieldGroundMudPhysics.installResetHooks()
	
	self:initEraseFoliageCache()
	
	if g_messageCenter ~= nil and MessageType ~= nil and MessageType.VEHICLE_RESET ~= nil then
		self._onVehicleReset = function(oldVehicle, newVehicle)
			if self.onVehicleReset ~= nil then
				self:onVehicleReset(oldVehicle, newVehicle)
			end
		end
		g_messageCenter:subscribe(MessageType.VEHICLE_RESET, self._onVehicleReset)
	end
	
	if g_server ~= nil or g_client ~= nil then
		if FieldGroundWheelRadiusSyncEvent ~= nil then
			if self.debug then
				print("[FieldGroundMudPhysics] FieldGroundWheelRadiusSyncEvent ready")
			end
		end
	end

	fgEnsureWheelIndices()
    addConsoleCommand("gsFieldMud", "FieldGroundMudPhysics on/off/debug", "consoleCommand", self)
end

function FieldGroundMudPhysics:deleteMap()
    removeConsoleCommand("gsFieldMud")

    if self._extraPS ~= nil then
        if self._extraPS.referencePS ~= nil then
            mpSafeDelete(self._extraPS.referencePS.shape)
            self._extraPS.referencePS = nil
        end
        if self._extraPS.referenceShape ~= nil then
            mpSafeDelete(self._extraPS.referenceShape)
            self._extraPS.referenceShape = nil
        end
    end
	
	FieldGroundMudPhysics.deleteResetHooks()
	
	if g_messageCenter ~= nil and self._onVehicleReset ~= nil and MessageType ~= nil and MessageType.VEHICLE_RESET ~= nil then
		g_messageCenter:unsubscribe(MessageType.VEHICLE_RESET, self._onVehicleReset)
		self._onVehicleReset = nil
	end

end

function FieldGroundMudPhysics:consoleCommand(arg)
    if arg == "on" then
        self.enabled = true
    elseif arg == "off" then
        self.enabled = false
    elseif arg == "debug" then
        self.debug = not self.debug
    end

    return string.format("FieldGroundMudPhysics enabled=%s debug=%s",
        tostring(self.enabled), tostring(self.debug))
end
