-- Author: U_BMP
-- Group: https://vk.com/biomodprod_utilit_fs
-- Date: 19.11.2025

MudWetnessMiniHUD = {}
MudWetnessMiniHUD.name = g_currentModName or "MudWetnessMiniHUD"
MudWetnessMiniHUD.path = g_currentModDirectory or ""

addModEventListener(MudWetnessMiniHUD)

-- =========================================================
-- SETTINGS
-- =========================================================

MudWetnessMiniHUD.enabled = true
MudWetnessMiniHUD.debug   = false

MudWetnessMiniHUD.wetIconRel  = "gui/wet.dds"
MudWetnessMiniHUD.rainIconRel = "gui/rain.dds"

MudWetnessMiniHUD.iconSizePx     = 17
MudWetnessMiniHUD.textPadPx      = 5
MudWetnessMiniHUD.roundToPercent = true  -- показывать 0..100%

-- GameInfoDisplay
MudWetnessMiniHUD.gapAboveGameInfoPx = 6

-- NEW: manual offsets (pixels)
--  offsetXPx: + -> left,  - -> right
--  offsetYPx: + -> up,    - -> down
MudWetnessMiniHUD.offsetXPx = 15
MudWetnessMiniHUD.offsetYPx = 0

-- visuals
MudWetnessMiniHUD.bgEnable = true
MudWetnessMiniHUD.bgAlpha  = 0.45

-- =========================================================
-- UTILS
-- =========================================================

local function clamp(v, a, b)
    if v < a then return a end
    if v > b then return b end
    return v
end

local function getWeather()
    if g_currentMission ~= nil
        and g_currentMission.environment ~= nil
        and g_currentMission.environment.weather ~= nil then
        return g_currentMission.environment.weather
    end
    return nil
end

local function getWetAndRain()
    local weather = getWeather()
    if weather == nil then
        return 0, 0
    end

    local wet = 0
    if weather.getGroundWetness ~= nil then
        wet = weather:getGroundWetness() or 0
    end

    local rain = 0
    if weather.getRainFallScale ~= nil then
        rain = weather:getRainFallScale() or 0
    end

    return clamp(wet, 0, 1), clamp(rain, 0, 1)
end

local function fmt01(v, asPercent)
    if asPercent then
        return string.format("%d%%", math.floor(v * 100 + 0.5))
    end
    return string.format("%.2f", v)
end

-- =========================================================
-- LIFECYCLE
-- =========================================================

function MudWetnessMiniHUD:loadMap()
    self.wetIconPath  = Utils.getFilename(self.wetIconRel,  self.path)
    self.rainIconPath = Utils.getFilename(self.rainIconRel, self.path)

    self:_rebuildOverlays()

    addConsoleCommand("gsMudWetMini", "MudWetnessMiniHUD on/off/toggle/debug", "consoleCommand", self)
end

function MudWetnessMiniHUD:deleteMap()
    removeConsoleCommand("gsMudWetMini")
    self:_deleteOverlays()
end

function MudWetnessMiniHUD:consoleCommand(arg)
    if arg == "on" then
        self.enabled = true
    elseif arg == "off" then
        self.enabled = false
    elseif arg == "toggle" or arg == nil or arg == "" then
        self.enabled = not self.enabled
    elseif arg == "debug" then
        self.debug = not self.enabled and false or (not self.debug)
    end

    local wet, rain = getWetAndRain()
    return string.format("MudWetnessMiniHUD enabled=%s debug=%s wet=%s rain=%s",
        tostring(self.enabled), tostring(self.debug), fmt01(wet, true), fmt01(rain, true))
end

function MudWetnessMiniHUD:_deleteOverlays()
    if self.wetIcon ~= nil then self.wetIcon:delete(); self.wetIcon = nil end
    if self.rainIcon ~= nil then self.rainIcon:delete(); self.rainIcon = nil end
    if self.bgOverlay ~= nil then self.bgOverlay:delete(); self.bgOverlay = nil end
end

function MudWetnessMiniHUD:_rebuildOverlays()
    self:_deleteOverlays()

    self.wetIcon  = Overlay.new(self.wetIconPath,  0, 0, 0.01, 0.01)
    self.rainIcon = Overlay.new(self.rainIconPath, 0, 0, 0.01, 0.01)

    if self.bgEnable then
        self.bgOverlay = Overlay.new(nil, 0, 0, 0.01, 0.01)
        self.bgOverlay:setColor(0, 0, 0, self.bgAlpha or 0.45)
    end
end

-- =========================================================
-- DRAW
-- =========================================================

function MudWetnessMiniHUD:draw()
    if not self.enabled or g_currentMission == nil or g_currentMission.hud == nil then
        return
    end

    local hud = g_currentMission.hud

    if hud.getIsVisible ~= nil then
        if not hud:getIsVisible() then
            return
        end
    elseif hud.isVisible ~= nil then
        if not hud.isVisible then
            return
        end
    end

    local gid = hud.gameInfoDisplay
    if gid == nil or gid.getPosition == nil then
        return
    end

    if gid.getVisible ~= nil and not gid:getVisible() then
        return
    end

    local posX, posY = gid:getPosition()

    local sW = gid.scalePixelToScreenWidth  ~= nil and function(px) return gid:scalePixelToScreenWidth(px) end or function(px) return px / 1920 end
    local sH = gid.scalePixelToScreenHeight ~= nil and function(px) return gid:scalePixelToScreenHeight(px) end or function(px) return px / 1080 end

    local iconSz = sH(self.iconSizePx or 18)
    local padX   = sW(self.textPadPx or 6)
    local gapX   = sW(10)
    local gapY   = sH(self.gapAboveGameInfoPx or 6)

    local offX = sW(self.offsetXPx or 0)
    local offY = sH(self.offsetYPx or 0)

    local textSize = gid.clockTextSize or gid.moneyTextSize or sH(17)

    local wet, rain = getWetAndRain()
    local wetText  = fmt01(wet,  self.roundToPercent == true)
    local rainText = fmt01(rain, self.roundToPercent == true)

    setTextAlignment(RenderText.ALIGN_LEFT)
    setTextBold(true)
    setTextColor(1, 1, 1, 1)

    local wetW  = iconSz + padX + getTextWidth(textSize, wetText)
    local rainW = iconSz + padX + getTextWidth(textSize, rainText)

    local blockW = wetW + gapX + rainW
    local blockH = iconSz

    local xRight = posX - offX
    local x0 = xRight - blockW
    local y0 = (posY + gapY) + offY

    if self.bgOverlay ~= nil then
        self.bgOverlay:setPosition(x0 - sW(6), y0 - sH(4))
        self.bgOverlay:setDimension(blockW + sW(12), blockH + sH(8))
        self.bgOverlay:render()
    end

    self.wetIcon:setPosition(x0, y0)
    self.wetIcon:setDimension(iconSz, iconSz)
    self.wetIcon:render()
    renderText(x0 + iconSz + padX, y0 + sH(2), textSize, wetText)

    local rx = x0 + wetW + gapX
    self.rainIcon:setPosition(rx, y0)
    self.rainIcon:setDimension(iconSz, iconSz)
    self.rainIcon:render()
    renderText(rx + iconSz + padX, y0 + sH(2), textSize, rainText)

    if self.debug then
        setTextBold(false)
        renderText(x0, y0 - sH(12), sH(12), string.format("wet=%.2f rain=%.2f", wet, rain))
        setTextBold(true)
    end

    setTextBold(false)
end
