--[[
Seasonal Prices adds seasonal variation to the prices of various products, both when buying and selling. 

Author:     w33zl / WZL Modding (https://github.com/w33zl)
Version:    2.0.0
Modified:   2025-01-02

Changelog:
    v2.0.0  - Updated to support FS25
    v1.0.0  - Initial FS22 release

]]

-- Init the mod using the "mod bootstrapper" from WeezlsModLib. Check https://github.com/w33zl/FS22_WeezlsModLib for details about the "Mod" class

local SeasonalPrices = Mod:init()
_G.SeasonalPrices = SeasonalPrices

local ENABLE_BUYSTATION = true
local USE_NEW_METHOD = false --TODO: this makes prices way to high, if enabled we should also change the base price of DIESEL and DEF
local ENABLE_BUGFIX_SELLSTATION = not USE_NEW_METHOD
local ENABLE_RANDOM_STATION_FACTOR = true

function SeasonalPrices:loadMap()
    ENABLE_RANDOM_STATION_FACTOR = ENABLE_RANDOM_STATION_FACTOR and not self:getIsMultiplayer() --HACK: simple fix to disable buy station randomness in MP
end


function SeasonalPrices:startMission()
    -- Log:debug("startMission")
    if self.onHourChanged ~= nil then
        self:onHourChanged()
    end
end

if ENABLE_RANDOM_STATION_FACTOR then
    Log:info("Random buying station prices enabled")

    local RANDOM_LOWER = 0.95
    local RANDOM_UPPER = 1.05
    local RANDOM_STEPS = (RANDOM_UPPER - RANDOM_LOWER) / 5 -- Smoothing factor
    local RANDOM_TOLERANCE = 0.0005
    local RANDOM_THRESHOLD_MAX = RANDOM_UPPER + RANDOM_TOLERANCE
    local RANDOM_THRESHOLD_MIN = RANDOM_LOWER - RANDOM_TOLERANCE

    local function getRandomFactor(previousValue)
        local currentDirection = -1
        
        if previousValue > RANDOM_UPPER then
            currentDirection = -1
        elseif previousValue < RANDOM_LOWER then
            currentDirection = 1
        else
            math.random() -- Make sure the random generator is in a new state
            currentDirection = math.random(-1, 1)
        end

        math.random() -- Make sure the random generator is in a new state

        local baseValue = math.pow(math.random(), 2)
        local relativeValue = RANDOM_STEPS * baseValue
        local actualDelta = relativeValue * currentDirection

        local newValue = previousValue + actualDelta

        newValue = math.min(newValue, RANDOM_THRESHOLD_MAX)
        newValue = math.max(newValue, RANDOM_THRESHOLD_MIN)

        return newValue, actualDelta
    end

    function SeasonalPrices:updateRandomBuyStationFactor(loadingStation)
        local owningPlaceable = loadingStation.owningPlaceable
        local spec = owningPlaceable and owningPlaceable.spec_buyingStation
        local buyingStation = spec and spec.buyingStation
        local isBuyingStation = buyingStation ~= nil

        if not owningPlaceable then
            return
        end

        -- local name = (owningPlaceable.getName and owningPlaceable:getName()) or "unknown"

        if not isBuyingStation then
            -- Log:debug("Not a buying station, skipping: #%s '%s' [%s], ", owningPlaceable.id, name, owningPlaceable.typeName)
            return
        end

        -- Log:debug("Checking station '%s' [%s], type name '%s', is buy station: %s", name, owningPlaceable.id, owningPlaceable.typeName, tostring(isBuyingStation))


        local currentRandomStationFactor = buyingStation.randomStationFactor or 1
        -- local previousRandomStationFactor = buyingStation.previousStationFactor or 1
        -- local currentDirection = (currentRandomStationFactor < previousRandomStationFactor) and -1 or 1
        local newRandomStationFactor = getRandomFactor(currentRandomStationFactor)
        buyingStation.previousStationFactor = currentRandomStationFactor
        buyingStation.randomStationFactor = newRandomStationFactor
    end

    function SeasonalPrices:onHourChanged()
        -- Log:debug("onHourChanged")
        if g_currentMission:getIsServer() then

            for _, loadingStation in pairs(g_currentMission.storageSystem.loadingStations) do
                self:updateRandomBuyStationFactor(loadingStation)
            end

        end
    end
end

if ENABLE_BUYSTATION then

    Log:info("Seasonal prices at buying stations enabled")

    if USE_NEW_METHOD then
        Log:debug("Using new method")
    end

    BuyingStation.getEffectiveFillTypePrice = Utils.overwrittenFunction(BuyingStation.getEffectiveFillTypePrice, function(self, superFunc, fillTypeIndex)
        local defaultStationPricePerLiter = superFunc(self, fillTypeIndex)
        local period, alpha = g_currentMission.environment:getPeriodAndAlphaIntoPeriod()
        local seasonalPriceFactor = g_currentMission.economyManager:getFillTypeSeasonalFactor(g_fillTypeManager:getFillTypeByIndex(fillTypeIndex), period, alpha)
        -- local difficultyModeFactor = EconomyManager.getPriceMultiplier()
        local buyStationBaseFactor = self.randomStationFactor or 1 -- Random factor per buy station
        local buyStationFillTypeFactor = self.fillTypePricesScale[fillTypeIndex]

        local function method1()
            -- Log:debug("method1")
            local pricePerLiterTmp = defaultStationPricePerLiter -- Has base price, station factor and difficulty factor (and difficulty factor if not DIESEL or DEF)
            pricePerLiterTmp = pricePerLiterTmp * seasonalPriceFactor -- Add seasonal factor
            return pricePerLiterTmp
        end

        local function method2()
            -- Log:debug("method2")
            local pricePerLiterTmp = g_currentMission.economyManager:getPricePerLiter(fillTypeIndex, true) -- Has seasonal factor, base price and difficulty factor
            return pricePerLiterTmp * buyStationFillTypeFactor -- Add buy station factor
        end


        return (USE_NEW_METHOD and method2() or method1()) * buyStationBaseFactor
    end)
end

if ENABLE_BUGFIX_SELLSTATION then
    Log:info("Selling station bugfix enabled")
    
    SellingStation.getEffectiveFillTypePrice = Utils.overwrittenFunction(SellingStation.getEffectiveFillTypePrice, function(self, superFunc, index, ...)
        local returnValue = superFunc(self, index, ...)
        local needsBugfix = (index == FillType.DIESEL or index == FillType.DEF)
        -- Log:debug("SellingStation.getEffectiveFillTypePrice: FillType: %s, needsBugfix: %s", g_fillTypeManager:getFillTypeNameByIndex(index), tostring(needsBugfix))
        return (needsBugfix and (returnValue / EconomyManager.getPriceMultiplier())) or returnValue
    end)
end

