-- SettingsGuiExt
-- Class to easily extend the settings gui
-- Version 1.0.0.0
-- @author PeterPwn247
-- Feel free to use it in your own mods

local modName = g_currentModName

SettingsGuiExt = {}

local TYPE = {
    AUTO = 1,
    HEADER = 2,
    BINARY = 3,
    MULTITEXT = 4,
    TEXT = 5,
    BUTTON = 6
}
SettingsGuiExt.TYPE = TYPE

local SettingsGuiExt_mt = Class(SettingsGuiExt)
local Settings_mt = {
    __index = function(t, k)
        local setting = rawget(getmetatable(t), k)
        if setting == nil then return nil end
        return rawget(setting, "value")
    end,
    __newindex = function(t, k, v)
        local setting = rawget(getmetatable(t), k)
        if setting == nil then return end
        if setting.state ~= nil then
            local state = rawget(rawget(setting, "optionsValueToIndex"), v)
            if state == nil then return end
            setting.state = state
        end
        rawset(setting, "value", v)
    end
}
local Setting_mt = {
    __index = function(t, k)
        if k == "state" then
            return rawget(t, "_state")
        end
        return nil
    end,
    __newindex = function(t, k, v)
        if k == "state" then
            local values = rawget(t, "optionsValues")
            if values == nil then return end
            local value = rawget(values, v)
            if value == nil then return end
            rawset(t, "_state", v)
            rawset(t, "value", value)
            return
        end
        rawset(t, k, v)
    end
}

function SettingsGuiExt.new(settings, layout, xmlPath, xmlFilename, baseKey)
    if settings == nil then return nil end
    if layout == nil then layout = g_inGameMenu.pageSettings.generalSettingsLayout end
    if layout.target == nil or not layout.target:isa(InGameMenuSettingsFrame) then return nil end
    if xmlPath == nil then xmlPath = g_modSettingsDirectory end
    if xmlFilename == nil then xmlFilename = modName:gsub("^FS25_", "") .. ".xml" end
    if baseKey == nil then baseKey = "settings" end

    local self = setmetatable({}, SettingsGuiExt_mt)
    self.settings = setmetatable(settings, Settings_mt)

    self.layout = layout
    self.name = layout.target.name
    self.xmlFilename = Utils.getFilename(xmlFilename, xmlPath)
    self.baseKey = baseKey
    self.elements = {}

    for _, setting in pairs(settings) do
        if type(setting) ~= "table" then continue end
        setmetatable(setting, Setting_mt)

        if setting.type == nil or setting.type == TYPE.AUTO then
            if setting.id == nil then
                setting.type = TYPE.HEADER
            elseif setting.options == nil or #setting.options == 2 then
                setting.type = TYPE.BINARY
            elseif #setting.options > 2 then
                setting.type = TYPE.MULTITEXT
            else
                setting.type = TYPE.TEXT
            end
        end

        if type(setting.id) == "string" then
            if setting.title == nil then setting.title = g_i18n:getText(setting.id .. "_title", modName) end
            if setting.toolTip == nil then setting.toolTip = g_i18n:getText(setting.id .. "_toolTip", modName) end
            if setting.disabled ~= nil and setting.disabledToolTip == nil then setting.disabledToolTip = g_i18n:getText(setting.id .. "_disabledToolTip", modName) end

            if setting.type == TYPE.BINARY and setting.options == nil then
                setting.options = {{false}, {true}}
            end

            if (setting.type == TYPE.BINARY or setting.type == TYPE.MULTITEXT) and setting.options ~= nil then
                setting.optionsValues = {}
                setting.optionsValueToIndex = {}
                setting.optionsTexts = {}

                local hasText = false
                for idx, option in pairs(setting.options) do
                    if option[1] == nil then continue end

                    table.insert(setting.optionsValues, option[1])
                    setting.optionsValueToIndex[option[1]] = idx
                    table.insert(setting.optionsTexts, option[2] or "")

                    hasText = hasText or option[2] ~= nil
                end
                if not hasText then setting.optionsTexts = nil end

                setting.state = setting.default or 1
            elseif setting.type == TYPE.TEXT and setting.value == nil then
                setting.value = setting.default or ""
            elseif setting.type == TYPE.BUTTON and setting.text == nil then
                setting.text = ""
            end

            Settings_mt[setting.id] = setting
        end
    end

    self.layout.target.onFrameClose = Utils.appendedFunction(self.layout.target.onFrameClose, function() self:writeSettings() end)

    self.xmlSchema = XMLSchema.new("SettingsGuiExt_" ..  baseKey)
    self:registerXMLPaths(self.xmlSchema)

    self:initialize()

    return self
end

function SettingsGuiExt:registerXMLPaths(schema)
    for _, setting in ipairs(self.settings) do
        if type(setting) ~= "table" or type(setting.id) ~= "string" then continue end

        local value = setting.state or setting.value
        if value == nil then continue end

        local key = string.format("%s.%s", self.baseKey, setting.id)

        local valueType
        if type(value) == "boolean" then
            valueType = XMLValueType.BOOL
        elseif type(value) == "number" then
            if MathUtil.isInt(value) then
                valueType = XMLValueType.INT
            else
                valueType = XMLValueType.FLOAT
            end
        else
            valueType = XMLValueType.STRING
        end

        schema:register(valueType, key)
    end
end

function SettingsGuiExt:readSettings()
    if not fileExists(self.xmlFilename) then
        self:writeSettings()
    end

    local xmlFile = XMLFile.loadIfExists("SettingsGuiExt", self.xmlFilename, self.xmlSchema)
    if xmlFile == nil then
        Logging.xmlWarning(self.xmlFilename, "Failed to read settings file.")
        return
    end

    for _, setting in ipairs(self.settings) do
        if type(setting) ~= "table" or type(setting.id) ~= "string" or setting.value == nil then continue end

        local key = string.format("%s.%s", self.baseKey, setting.id)

        local value = xmlFile:getValue(key)
        if value == nil then continue end

        if setting.state ~= nil then
            value = math.clamp(value, 1, #setting.options)
            setting.state = value
        else
            if setting.validate ~= nil and type(setting.validate) == "function" then
                value = setting.validate(self, setting, value)
            end
            setting.value = value
        end
    end

    xmlFile:delete()
end

function SettingsGuiExt:writeSettings()
    local xmlFile = XMLFile.loadIfExists("SettingsGuiExt", self.xmlFilename, self.xmlSchema)
    if xmlFile == nil then
        xmlFile = XMLFile.create("SettingsGuiExt", self.xmlFilename, self.baseKey, self.xmlSchema)
    end
    if xmlFile == nil then
        Logging.xmlWarning(self.xmlFilename, "Failed to write settings file.")
        return
    end

    for _, setting in ipairs(self.settings) do
        if type(setting) ~= "table" or type(setting.id) ~= "string" then continue end

        local value = setting.state or setting.value
        if value == nil then continue end

        local key = string.format("%s.%s", self.baseKey, setting.id)

        xmlFile:setValue(key, value)
        -- on dedicated server add comment with valid options
    end

    xmlFile:save()
    xmlFile:delete()
end

function SettingsGuiExt:initialize()
    self:readSettings()
    self:extendGui()
end

function SettingsGuiExt:extendGui()
    FocusManager:setGui(self.name)

    local guiXML = ""
    for _, setting in ipairs(self.settings) do
        if type(setting) ~= "table" then continue end

        local elementXML = ""
        if setting.type == TYPE.HEADER then
            elementXML = [[<Text profile="fs25_settingsSectionHeader" name="sectionHeader" text="$setting_title"/>]]
        elseif setting.type == TYPE.BINARY then
            elementXML = [[<Bitmap profile="fs25_multiTextOptionContainer">
                    <BinaryOption profile="fs25_settingsBinaryOption" id="$setting_id" onCreate="onCreate" onOpen="onOpen" onClose="onClose" onClick="onSetState">
                        <Text profile="fs25_multiTextOptionTooltip" name="ignore" text="$setting_toolTip"/>
                    </BinaryOption>
                    <Button profile="fs25_settingsMultiTextOptionLocked" name="iconDisabled" onClick="NO_CALLBACK">
                        <Text profile="fs25_multiTextOptionTooltip" text="$setting_disabledToolTip" position="940px 0px"/>
                    </Button>
                    <Text profile="fs25_settingsMultiTextOptionTitle" text="$setting_title"/>
                </Bitmap>]]
        elseif setting.type == TYPE.MULTITEXT then
            elementXML = [[<Bitmap profile="fs25_multiTextOptionContainer">
                    <MultiTextOption profile="fs25_settingsMultiTextOption" id="$setting_id" onCreate="onCreate" onOpen="onOpen" onClose="onClose" onClick="onSetState">
                        <Text profile="fs25_multiTextOptionTooltip" name="ignore" text="$setting_toolTip"/>
                    </MultiTextOption>
                    <Button profile="fs25_settingsMultiTextOptionLocked" name="iconDisabled" onClick="NO_CALLBACK">
                        <Text profile="fs25_multiTextOptionTooltip" text="$setting_disabledToolTip" position="940px 0px"/>
                    </Button>
                    <Text profile="fs25_settingsMultiTextOptionTitle" text="$setting_title"/>
                </Bitmap>]]
        elseif setting.type == TYPE.TEXT then
            elementXML = [[<Bitmap profile="fs25_multiTextOptionContainer">
                    <TextInput profile="fs25_settingsTextInput" id="$setting_id" onCreate="onCreate" onOpen="onOpen" onClose="onClose" onLeave="onLeave" focusOnHighlight="true" imeDescription="$setting_title" imePlaceholder="$setting_title" imeTitle="$setting_title">
                        <ThreePartBitmap profile="fs25_textInputBg"/>
                        <Bitmap profile="fs25_textInputIconBox">
                            <Bitmap profile="fs25_textInputIcon"/>
                        </Bitmap>
                        <Text profile="fs25_multiTextOptionTooltip" name="ignore" text="$setting_toolTip" position="678px 0px"/>
                    </TextInput>
                    <Button profile="fs25_settingsMultiTextOptionLocked" name="iconDisabled" onClick="NO_CALLBACK">
                        <Text profile="fs25_multiTextOptionTooltip" text="$setting_disabledToolTip" position="940px 0px"/>
                    </Button>
                    <Text profile="fs25_settingsMultiTextOptionTitle" text="$setting_title"/>
                </Bitmap>]]
        elseif setting.type == TYPE.BUTTON then
            elementXML = [[<Bitmap profile="fs25_multiTextOptionContainer">
                    <Button profile="fs25_settingsHDRButtonConsole" id="$setting_id" onCreate="onCreate" onOpen="onOpen" onClose="onClose" onClick="onClick" text="$setting_text" iconSliceId="controllerSymbols.mouse_button_Left">
                        <ThreePartBitmap profile="fs25_settingsButtonBg"/>
                        <Text profile="fs25_multiTextOptionTooltip" name="ignore" text="$setting_toolTip"/>
                    </Button>
                    <Button profile="fs25_settingsMultiTextOptionLocked" name="iconDisabled" onClick="NO_CALLBACK">
                        <Text profile="fs25_multiTextOptionTooltip" text="$setting_disabledToolTip" position="940px 0px"/>
                    </Button>
                    <Text profile="fs25_settingsMultiTextOptionTitle" text="$setting_title"/>
                </Bitmap>]]
            -- focus fails if last in menu
        else
            continue
        end

        guiXML = guiXML .. elementXML:gsub("%$setting_([%a_]+)", setting)
    end

    local xmlFile = loadXMLFileFromMemory("SettingsGuiExt", "<GUI>" .. guiXML .. "</GUI>")
    g_gui:loadGuiRec(xmlFile, "GUI", self.layout, self)
    delete(xmlFile)

    self.layout.target:updateAbsolutePosition()
    self.layout.target:onGuiSetupFinished()
end

function SettingsGuiExt:onCreate(element)
    element:onGuiSetupFinished()

    if element.id == nil then return end

    local setting = getmetatable(self.settings)[element.id]
    if setting == nil then return end

    self.elements[setting.id] = element

    if setting.type == TYPE.BINARY or setting.type == TYPE.MULTITEXT then
        if setting.optionsTexts ~= nil then
            element:setTexts(setting.optionsTexts)
        end
        element:setState(setting.state)
    elseif setting.type == TYPE.TEXT then
        element:setText(setting.value)
    end

    if setting.onCreate ~= nil and type(setting.onCreate) == "function" then
        setting.onCreate(self, setting, unpack(setting.onCreateArgs or {}))
    end
end

function SettingsGuiExt:onOpen(element)
    local disabled = self.layout:getIsDisabled()

    local setting = nil
    if element.id ~= nil then
        setting = getmetatable(self.settings)[element.id]
        if not disabled and setting ~= nil then
            if type(setting.disabled) == "function" then
                disabled = setting.disabled(self, setting)
            elseif setting.disabled ~= nil then
                disabled = setting.disabled
            end
        end
    end

    element.parent:setDisabled(disabled)
    local iconDisabled = element.parent:getDescendantByName("iconDisabled")
    if iconDisabled ~= nil then
        iconDisabled:setDisabled(not disabled)
    end

    if setting ~= nil and setting.onOpen ~= nil and type(setting.onOpen) == "function" then
        setting.onOpen(self, setting, unpack(setting.onOpenArgs or {}))
    end
end

function SettingsGuiExt:onClose(element)
    if element.id == nil then return end

    local setting = getmetatable(self.settings)[element.id]
    if setting == nil then return end

    if setting.onClose ~= nil and type(setting.onClose) == "function" then
        setting.onClose(self, setting, unpack(setting.onCloseArgs or {}))
    end
end

function SettingsGuiExt:onLeave(element)
    if element.id == nil then return end

    local setting = getmetatable(self.settings)[element.id]
    if setting == nil then return end

    if setting.type == TYPE.TEXT then
        local value = element:getText()
        if setting.validate ~= nil and type(setting.validate) == "function" then
            value = setting.validate(self, setting, value)
        end
        setting.value = value
    end

    if setting.onLeave ~= nil and type(setting.onLeave) == "function" then
        setting.onLeave(self, setting, unpack(setting.onLeaveArgs or {}))
    end
end

function SettingsGuiExt:onClick(element)
    if element == nil or element.id == nil then return end

    local setting = getmetatable(self.settings)[element.id]
    if setting == nil then return end

    if setting.onClick ~= nil and type(setting.onClick) == "function" then
        setting.onClick(self, element, unpack(setting.onClickArgs or {}))
    end
end

function SettingsGuiExt:onSetState(state, element)
    if element == nil or element.id == nil then return end

    local setting = getmetatable(self.settings)[element.id]
    if setting == nil then return end

    setting.state = state

    if setting.onClick ~= nil and type(setting.onClick) == "function" then
        setting.onClick(self, element, state, unpack(setting.onClickArgs or {}))
    end
end

function SettingsGuiExt.NO_CALLBACK() end

function SettingsGuiExt:disableInputForDuration(duration)
    self.layout.target:disableInputForDuration(duration)
end

function SettingsGuiExt:isInputDisabled()
    return self.layout.target:isInputDisabled()
end