local modName, modDir = g_currentModName, g_currentModDirectory
local modInfo = g_modManager:getModByName(modName)

source(Utils.getFilename("SettingsGuiExt.lua", modDir .. "scripts/"))
source(Utils.getFilename("EngineStartDisplay.lua", modDir .. "scripts/"))

EngineStartPlus = {}

function EngineStartPlus:init()
    EngineStartPlus:registerExtendedFunctions()

    local seconds, minutes = g_i18n:getText("seconds_format", modName), g_i18n:getText("minutes_format", modName)

    self.settings = {
        {
            title = modInfo.title
            -- type: without id defaults to TYPE.HEADER
        },
        {
            id = "isEnabled",
            default = 2
            -- type: without options or 2 options defaults to TYPE.BINARY
            -- title: l10n entry $id_title, here isEnabled_title
            -- toolTip: l10n entry $id_toolTip, here isEnabled_toolTip
            -- options: type TYPE.BINARY defaults to {{false}, {true}} -> will show off, on in gui
        },
        {
            id = "motorStopTimer",
            options = {
                {-1, g_i18n:getText("setting_off")},
                {10000, seconds:format(10)},
                {20000, seconds:format(20)},
                {30000, seconds:format(30)},
                {60000, seconds:format(60)},
                {90000, seconds:format(90)},
                {120000, minutes:format(2)},
                {180000, minutes:format(3)},
                {240000, minutes:format(4)},
                {300000, minutes:format(5)}
            },
            disabled = g_server == nil
            -- type: number of options is 2 -> TYPE.BINARY, else -> TYPE.MULTITEXT
            -- disabledToolTip: l10n entry $id_disabledToolTip, here motorStopTimer_disabledToolTip
        },
        {
            id = "turnOnHeadLights",
            default = 2
        },
        {
            id = "turnOnHighBeam",
          },
        {
            id = "turnOnWorkingLightsFront"
        },
        {
            id = "turnOnWorkingLightsBack"
        },
        {
            id = "turnOnLightsOnlyAtNight",
            default = 2
        },
        {
            id = "turnOnHazardLights"
        },
        {
            id = "turnOnBeaconLights"
        },
        {
            id = "turnOffLights",
            default = 2
        }
    }
    self.settingsGui = SettingsGuiExt.new(self.settings)

    if g_server ~= nil and self.settings.motorStopTimer ~= -1 then
        g_gameSettings:setValue(GameSettings.SETTING.MOTOR_STOP_TIMER_DURATION, self.settings.motorStopTimer)
    end
end

function EngineStartPlus:registerExtendedFunctions()
    local function prependFunction(object, funcName, func)
        object[funcName] = Utils.prependedFunction(object[funcName], function(...) func(self, ...) end)
    end
    local function appendFunction(object, funcName, func)
        object[funcName] = Utils.appendedFunction(object[funcName], function(...) func(self, ...) end)
    end
    local function overwriteFunction(object, funcName, func)
        object[funcName] = Utils.overwrittenFunction(object[funcName], function(...) func(self, ...) end)
    end

    appendFunction(BaseMission, "onFinishedLoading", self.missionLoaded)

    appendFunction(Motorized, "onUpdateTick", self.motorizedUpdateTick)
    appendFunction(Motorized, "startMotor", self.motorizedStartMotor)
    prependFunction(Motorized, "stopMotor", self.hideDisplay)

    prependFunction(Enterable, "onPlayerLeaveVehicle", self.hideDisplay)

    appendFunction(Lights, "onStartMotor", self.lightsStartMotor)
    prependFunction(Lights, "onStopMotor", self.lightsStopMotor)
    appendFunction(Lights, "lightsWeatherChanged", self.lightsWeatherChanged)

    overwriteFunction(Vehicle, "addPoweredActionEvent", self.vehicleAddActionEvent)

    prependFunction(Drivable, "actionEventAccelerate", self.startMotor)
    prependFunction(Drivable, "actionEventBrake", self.startMotor)
    prependFunction(Drivable, "actionEventCruiseControlState", self.startMotor)
    appendFunction(Drivable, "onLeaveVehicle", self.drivableLeaveVehicle)
end

function EngineStartPlus:missionLoaded(mission)
    if g_server ~= nil then
        mission.missionInfo.automaticMotorStartEnabled = false
    end

    local checkAutoMotorStart = g_inGameMenu.pageSettings.checkAutoMotorStart
    if checkAutoMotorStart ~= nil then
        local lockXML = [[<GUI>
                <Button profile="fs25_settingsMultiTextOptionLocked" name="iconDisabled">
                    <Text profile="fs25_multiTextOptionTooltip" text="$l10n_checkAutoMotorStart_disabled" position="940px 0px"/>
                </Button>
            </GUI>]]

        local xmlFile = loadXMLFileFromMemory("SettingsGuiExt", lockXML:gsub("%$l10n_([%a_]+)", function(s) return g_i18n:getText(s, modName):format(modInfo.title) end))
        g_gui:loadGuiRec(xmlFile, "GUI", checkAutoMotorStart.parent, g_inGameMenu.pageSettings)
        delete(xmlFile)

        local lockElement = checkAutoMotorStart.parent:getDescendantByName("iconDisabled")
        if lockElement ~= nil then
            lockElement.onClickCallback = function() end
            checkAutoMotorStart.onOpenCallback = Utils.prependedFunction(checkAutoMotorStart.onOpenCallback, function(page)
                page.checkAutoMotorStart.parent:setDisabled(true)
                lockElement:setDisabled(false)
            end)
        end
    end

    g_overlayManager:addTextureConfigFile(Utils.getFilename("engine.xml", modDir .. "icon/"), "gui", modName)
    self.engineStartDisplay = EngineStartDisplay.new()
end

function EngineStartPlus:motorizedUpdateTick(vehicle, dt)
    if g_server == nil or self.settings.motorStopTimer == -1 or vehicle:getIsAIActive() or vehicle.spec_locomotive ~= nil then return end

    local motorState = vehicle:getMotorState()
    if not (motorState == MotorState.ON or motorState == MotorState.STARTING) then return end

    if (vehicle.getIsControlled ~= nil and vehicle:getIsControlled()) or (vehicle.getIsEnteredForInput ~= nil and vehicle:getIsEnteredForInput()) then return end

    vehicle:raiseActive()
end

function EngineStartPlus:motorizedStartMotor(vehicle)
    local rootVehicle = vehicle:getRootVehicle()
    if rootVehicle == nil or not rootVehicle:getIsActiveForInput(true) then return end

    local name = vehicle:getName()
    if vehicle.brand ~= nil and vehicle.brand.title ~= nil and vehicle.brand.name ~= nil and utf8ToUpper(vehicle.brand.name) ~= "NONE" then
        name = vehicle.brand.title .. " " .. name
    end

    self.engineStartDisplay:show(vehicle.spec_motorized.motorStartDuration, name)
end

function EngineStartPlus:lightsStartMotor(lights)
    if lights:getCanToggleLight() then
        self:turnOnLights(lights)
        if self.settings.turnOnHazardLights then
            lights:setTurnLightState(Lights.TURNLIGHT_HAZARD)
        end
        if self.settings.turnOnBeaconLights then
            lights:setBeaconLightsVisibility(true)
        end
    end
end

function EngineStartPlus:lightsStopMotor(lights)
    if self.settings.turnOffLights then
        lights:deactivateLights()
    end
end

function EngineStartPlus:lightsWeatherChanged(lights)
    if not self.settings.isEnabled then return end

    local rootVehicle = lights:getRootVehicle()
    if rootVehicle == nil or rootVehicle.spec_motorized == nil or rootVehicle ~= g_localPlayer:getCurrentVehicle() then return end

    local motorState = rootVehicle:getMotorState()
    if motorState ~= MotorState.ON and motorState ~= MotorState.STARTING then return end

    if g_currentMission.environment.isSunOn then
        if self.settings.turnOnLightsOnlyAtNight and lights.spec_lights.lightsTypesMask ~= 0 then
            lights:setLightsTypesMask(0)
        end
    else
        self:turnOnLights(lights)
    end
end

function EngineStartPlus:vehicleAddActionEvent(vehicle, superFunc, actionEventsTable, inputAction, target, callback, ...)
    if self.settings.isEnabled then
        callback = Utils.prependedFunction(callback, function(...) self.startMotor(self, ...) end)
        return vehicle:addActionEvent(actionEventsTable, inputAction, target, callback, ...)
    end

    return superFunc(vehicle, actionEventsTable, inputAction, target, callback, ...)
end

function EngineStartPlus:drivableLeaveVehicle(vehicle)
    if vehicle.spec_motorized == nil then return end
    vehicle.spec_motorized.motorStopTimer = vehicle.spec_motorized.motorStopTimerDuration
end

function EngineStartPlus:startMotor(vehicle, actionName, inputValue)
    if not self.settings.isEnabled then return end
    if inputValue == 0 or actionName == InputAction.AXIS_MOVE_SIDE_VEHICLE then return end

    local rootVehicle = vehicle:getRootVehicle()
    if rootVehicle == nil or rootVehicle.spec_motorized == nil or not rootVehicle:getIsActiveForInput(true) then return end

    local motorState = rootVehicle:getMotorState()
    if motorState == MotorState.ON or motorState == MotorState.STARTING or not rootVehicle:getCanMotorRun() then return end

    rootVehicle:startMotor()
end

function EngineStartPlus:turnOnLights(vehicle)
    if not (self.settings.turnOnLightsOnlyAtNight and g_currentMission.environment.isSunOn) then
        if vehicle:getCanToggleLight() then
            local lightsTypesMask = vehicle.spec_lights.lightsTypesMask
            if self.settings.turnOnHeadLights then
                lightsTypesMask = bit32.bor(lightsTypesMask, 2 ^ Lights.LIGHT_TYPE_DEFAULT)
            end
            if self.settings.turnOnHighBeam then
                lightsTypesMask = bit32.bor(lightsTypesMask, 2 ^ Lights.LIGHT_TYPE_HIGHBEAM)
            end
            if self.settings.turnOnWorkingLightsFront then
                lightsTypesMask = bit32.bor(lightsTypesMask, 2 ^ Lights.LIGHT_TYPE_WORK_FRONT)
            end
            if self.settings.turnOnWorkingLightsBack then
                lightsTypesMask = bit32.bor(lightsTypesMask, 2 ^ Lights.LIGHT_TYPE_WORK_BACK)
            end
            if lightsTypesMask ~= vehicle.spec_lights.lightsTypesMask then
                vehicle:setLightsTypesMask(lightsTypesMask)
            end
        end
    end
end

function EngineStartPlus:hideDisplay()
    self.engineStartDisplay:hide()
end

EngineStartPlus:init()