--[[
    Keep me Admin Settings

    Adds the settings for the Keep me admin mod to the general settings menu.

        @author:    BayernGamers
        @date:      09.04.2025
        @version:   1.0

        History:    v1.0 @09.04.2025 - initial implementation in FS25
                    -------------------------------------------------------------------------------------------

        License:    Terms:
                        Usage:
                            Feel free to use this work as-is as long as you adhere to the following terms:
						Attribution:
							You must give appropriate credit to the original author when using this work.
						No Derivatives:
							You may not alter, transform, or build upon this work in any way.
						Usage:
							The work may be used for personal and commercial purposes, provided it is not modified or adapted.
						Additional Clause:
							This script may not be converted, adapted, or incorporated into any other game versions or platforms except by GIANTS Software.
]]
source(Utils.getFilename("scripts/utils/LoggingUtil.lua", g_currentModDirectory))

local log = LoggingUtil.new(true, LoggingUtil.DEBUG_LEVELS.HIGH, "KeepMeAdminSettings.lua")

KeepMeAdminSettings = {}
KeepMeAdminSettings.MOD_DIRECTORY = g_currentModDirectory
KeepMeAdminSettings.MOD_NAME = g_currentModName
KeepMeAdminSettings.MOD_SETTINGS_DIRECTORY = g_currentModSettingsDirectory

local KeepMeAdminSettings_mt = Class(KeepMeAdminSettings)

function KeepMeAdminSettings.new(keepMeAdmin)
    local self = setmetatable({}, KeepMeAdminSettings_mt)

    self.keepMeAdmin = keepMeAdmin

    return self
end

function KeepMeAdminSettings:init()
    local showDialog, enabled = self.keepMeAdmin:getUserSettings()
    local settingsLayout = g_inGameMenu.generalSettingsLayout

    self:addTitle(g_i18n:getText("title_keepMeAdmin"))

    local showDialogState = showDialog and 2 or 1
    self:addCheckBox("showLoadAdminPasswordDialog", g_i18n:getText("setting_showLoadAdminPasswordDialog"), g_i18n:getText("toolTip_showLoadAdminPasswordDialog"), showDialogState, "onShowDialogChanged")

    local enabledState = enabled and 2 or 1
    self:addCheckBox("enableLoadStoredAdminPassword", g_i18n:getText("setting_enableLoadStoredAdminPassword"), g_i18n:getText("toolTip_enableLoadStoredAdminPassword"), enabledState, "onEnabledChanged")

    settingsLayout:invalidateLayout()
end

function KeepMeAdminSettings:onShowDialogChanged()
    local showDialog, _ = self.keepMeAdmin:getUserSettings()
    log:printDevInfo("Setting showDialog to " .. tostring(not showDialog), LoggingUtil.DEBUG_LEVELS.HIGH)
    self.keepMeAdmin:saveShowDialog(not showDialog)
end

function KeepMeAdminSettings:onEnabledChanged()
    local _, enabled = self.keepMeAdmin:getUserSettings()
    log:printDevInfo("Setting enabled to " .. tostring(not enabled), LoggingUtil.DEBUG_LEVELS.HIGH)
    self.keepMeAdmin:saveAutoLoadPassword(not enabled)
end

function KeepMeAdminSettings:updateElementFocusIds(element)
    if not element then
        return
    end
    element.focusId = FocusManager:serveAutoFocusId()
    for _, child in pairs(element.elements) do
        self:updateElementFocusIds(child)
    end
end

function KeepMeAdminSettings:addTitle(text)
    local settingsLayout = g_inGameMenu.generalSettingsLayout
    local controlsList = g_inGameMenu.controlsList

    local title = nil
    for _, element in ipairs(settingsLayout.elements) do
        if element.name == "sectionHeader" then
            title = element:clone(settingsLayout)
            title:setText(text)

            self:updateElementFocusIds(title)
            table.insert(controlsList, title)
            return
        end
    end
end

function KeepMeAdminSettings:addCheckBox(id, text, toolTip, state, callback)
    local settingsLayout = g_inGameMenu.generalSettingsLayout
    local checkBoxTemplate = g_inGameMenu.checkWoodHarvesterAutoCutBox
    local controlsList = g_inGameMenu.controlsList

    local checkBox = checkBoxTemplate:clone(settingsLayout)
    checkBox.id = id

    local checkBoxElement = checkBox.elements[1]
    checkBoxElement.id = id
    checkBoxElement.target = self

    checkBoxElement:setCallback("onClickCallback", callback)
    checkBoxElement:setDisabled(false)

    local toolTipElement = checkBoxElement.elements[1]
    toolTipElement:setText(toolTip)

    local settingElement = checkBox.elements[2]
    settingElement:setText(text)

    checkBoxElement:setTexts({g_i18n:getText("ui_no"), g_i18n:getText("ui_yes")})
    checkBoxElement:setState(state)

    self:updateElementFocusIds(checkBox)
    table.insert(controlsList, checkBox)
end